%--------------------------------------------------------------------------
% function [elapsed_time]=FT_print_progress(elapsed_time, nrun, total_runs, newline)
%
% A function for Matlab that prints a progress statement in a loop
%
% elapsed_time (real):   how many seconds have been spend in the loop [s]
% nrun (integer):        number of current iteration
% total_runs (integer):  total number of iterations
% newline (integer):     put newline=1 if a new line should be printed 
%                        before the next progress statement
%
% example use:
%
% total_runs=20
% elapsed_time=0;
% for i=1:total_runs
%   tic;
%   newline=0;
%   do_some_stuff();
%   if (something_happens)
%     print_warning;
%     newline=1;
%   end
%   [elapsed_time]=FT_print_progress(elpased_time,i,total_runs, newline);
% end;
%
% Andreas Freise, Simon Chelkowski  14.11.2009
%--------------------------------------------------------------------------
%

% Description: Prints a progress statement in a loop
% Keywords: progress

function [elapsed_time]=FT_print_progress(elapsed_time, nrun, total_runs, newline)

  ntime = toc; % readout timer
  
  if ((elapsed_time~=0) && (newline==0))
    fprintf(['%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c'] ...
            ,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8);
  end
  if(newline)
    fprintf('\n');
  end
  
  elapsed_time = elapsed_time + ntime;
  ave_run_time = elapsed_time / nrun;
  estimated_time = (total_runs - nrun + 2) * ave_run_time;
  if (nrun==total_runs)
    fprintf([' %3d%%  -  %02.0f:%02.0f:%02.0f elapsed time\n'] ...
            ,100,fix(elapsed_time/3600),fix(mod(elapsed_time,3600)/60),fix(mod(mod(elapsed_time,3600),60)));
  else
    fprintf([' %3d%%  -  %02.0f:%02.0f:%02.0f remaining!'] ...
            ,round(nrun/total_runs*100),fix(estimated_time/3600),fix(mod(estimated_time,3600)/60),fix(mod(mod(estimated_time,3600),60)));
   end
end