%
%--------------------------------------------------------------------------
% function [power]=FT_cavity_throughput(T_in, T_out, F, detuning)
%
% A function for Matlab which computes the throughput of a cavity
% (normalises transmitted power) as a function of the power transmittance
% of the input and output mirror and the cavity's finesse.
%
% T_in (real):     power transmittance of input mirror
% T_in (real):     power transmittance of input mirror
% F (real):        cavity finesse
% detuning (real): detuning of the cavity from 0 to pi 
%
% power (real):    throughput P_out/P_in
%
% Part of the SimTools package
% Andreas Freise 08.02.2010 afreise@googlemail.com
%--------------------------------------------------------------------------
%

% Description: Computes the throughput of a cavity as a function of power
% Description: transmittance and cavity finesse
% Keywords: throughput, transmittance

function [power]=FT_cavity_throughput(T_in, T_out, F, detuning)
  
  verbose=0;
  F=abs(F);
  % compute r_1 * r_2 from Finesse
  tmp1=sin(pi/2.0/F);
  tmp2=-tmp1+sqrt(1+tmp1^2);
  r1r2=tmp2^2;
  
  % check if r1r2 is too high
  rrrr=sqrt(1-T_in)*sqrt(1-T_out);
  if (r1r2>rrrr && verbose==1)
    error('Finesse is too large for given T_in, T_out.');
  end
  
  % airy function for amplitude
  t1=sqrt(T_in);
  t2=sqrt(T_out);
  x=detuning;
  
  airy=-t1*t2*exp(-i*x)./(1-r1r2*exp(-i*2*x));
  
  power=abs(airy).^2;
  
  