%
% --------------------------------------------------------------
% function [a]= FT_cavity_fields(D, R1, T1, R2, T2, lambda, f_offset)
%
% Matlab function to compute the fields of a two mirror cavity
%
% ---> a_0   | --> a_1               | --> a_2
% <--- a_4   |             a_3 <---  |   
%            mirror1              mirror2
%
% D: cavity length either as 
%    - one real value (length in meters)
%    - a vector [L, phi] with two real values
%      - L macroscopic length in meters
%      - microscopic tuning in degrees (with 360 deg = lambda)
% R1, R2 (real):  power reflectivity of mirror1, mirror2
% T1, T2 (real):  power transmittance of mirror1, mirror2
% lambda (real):  wavelength of the laser
% f_offset (real): frequency offset of the laser light (i.e. f=c/lambda+f_offset)
%
% a (complex): vector with cavity fields as shown in above image:
%     - a(1)=a_1/a_0:  internal field leaving mirror1
%     - a(2)=a_2/a_0: transmitted field
%     - a(3)=a_3/a_0: internal field leaving mirror2
%     - a(4)=a_4/a_0: reflected field
%
% Part of the Simtools package
% Andreas Freise 18.08.2009 afreise@googlemail.com
% --------------------------------------------------------------
%

% Description: Computes the fields of a two mirror cavity
% Keywords: field, two, mirror

function [a]= FT_cavity_fields(D, R1, T1, R2, T2, lambda, f_offset)

  c=299792458.0;
  w0=2*pi*c/lambda;
  delta_w=2*pi*f_offset;
  w=w0+delta_w;
 
  [rows,cols]=size(D);
  if (cols==2)
    kD = D(:,1)*delta_w./c+w./w0*D(:,2)/180*pi;  
  else
    kD = D*w/c;
  end
  
  exp1=exp(-i * kD);
  exp2=exp1.*exp1;
  
  r1=sqrt(R1);
  r2=sqrt(R2);
  t1=sqrt(T1);
  t2=sqrt(T2);
  
  d=1./(1-r1.*r2.*exp2);
  a(1,:)=i*t1.*d;
  a(2,:)=-t1.*t2.*exp1.*d;
  a(3,:)=i*r2.*t1.*exp1.*d;
  a(4,:)=r1-r2.*t1.^2.*exp2.*d;
  
  return;
  
  
