%
%---------------------------------------------------------------------  
% function [mirror] = FT_add_mirror_map(mirror, mygrid, lambda)
% 
% Matlab function that computes mirror maps for reflection and
% transmission of a mirror. The maps are added to the structure 
% `mirror', see FT_init_mirror.m for more info.
%
% mirror (structure):  stores the parameters of a mirror component
%                      see FT_init_mirror.m
% mygrid (structure):  stores parameters of a 2D grid for FFT computations,
%                      see FT_init_grid.m
% lambda (real):       wavelength of laser [m]
%
% Part of the SimTools package
% Andreas Freise, Jerome Degallaix 26.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  
%

% Description: Computes mirror maps for reflection and transmission of a mirror
% Keywords: add, mirror, map

function [mirror] = FT_add_mirror_map(mirror, mygrid, lambda)

  % Initialising map grid 
  mirror.map_trans = complex(zeros(mygrid.xpoints,mygrid.ypoints,'double'));
  mirror.map_refl_back  =  mirror.map_trans;
  mirror.map_refl_front  =  mirror.map_trans;
    
  % compute aperture map
  mask_idx=find(mygrid.r<mirror.diameter/2.0);
  mirror_mask=mirror.map_trans;
  mirror_mask(mask_idx)=1.0;
  
  map_base=zeros(mygrid.xpoints,mygrid.ypoints,'double');
  
  if (~isinf(mirror.Rc(1)))
    %%% TODO change this to use Rcx, Rcy!
    map_base = (mirror.Rc(1) - sign(mirror.Rc(1))*sqrt(mirror.Rc(1)^2 - mygrid.r_squared));
  end
  
  % Store original amplitude and phase map in case we want to
  % multiply another map with this one later and then recalculate
  mirror.map_phase = map_base;
  mirror.map_amp   = mirror_mask;
  
  % tilt maps if necessary
  if (mirror.alpha~=0.0)
    map_base    =  FT_tilt_map(map_base,    mygrid, mirror.alpha);
    mirror_mask =  FT_tilt_map(mirror_mask, mygrid, mirror.alpha);
  end
  
  phase_refl_front =             -2.0 * map_base;
  phase_refl_back  =   2.0 * mirror.n * map_base;
  phase_trans      = (mirror.n - 1.0) * map_base;
  
  k=2*pi/lambda;

  mirror.map_refl_front = mirror_mask .*  exp(-1i * k * phase_refl_front);
  mirror.map_refl_back  = mirror_mask  .*  exp(-1i * k * phase_refl_back);
  mirror.map_trans      = mirror_mask      .*  exp(-1i * k * phase_trans);  
  
  return
  


  