%--------------------------------------------------------------------------
% function [Z] = FT_zernike_polynomial_for_grid(xpoints,ypoints,x0,y0,rx,ry,n,m)
%
% A function for Matlab which calculates surface heights defined by a
% Zernike polynomial for a particular grid.  The polynomials are defined by
% integer indices n and m.  Requires FT_Znm.m.
%
% xpoints/
% ypoints:      Size of the surface
% x0/y0:        Optical centre
% rx/ry:        Radius in x and y directions (in indices)
% n:            Radial index, n>=0
% m:            Azimuthal index.  For each value of n the values of m are 
%               from -n to +n, in steps of 2, i.e. -n, 2-n, 4-n .... n-2, n
%
% Z:            Returned Zernike polynomial surface heights
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    06.12.09
%--------------------------------------------------------------------------


function [Z] = FT_zernike_polynomial_for_grid(xpoints,ypoints,x0,y0,rx,ry,n,m)

    % Define surface size
    newx=(linspace(1,xpoints,xpoints)-x0);
    newy=(linspace(1,ypoints,ypoints)-y0);
    
    % Define radial and angular coordinates
    [X,Y]=meshgrid(newx/rx,newy/ry);
    phi=atan2(Y,X);
    rho=sqrt(X.^2+Y.^2);
    
    % Calculate Zernike surface heights
    Z = FT_Znm(n,m,rho,phi);

end

