%--------------------------------------------------------------------------
% function [zc] = FT_reshuffle_zernike_content(zc)
%
% A function for Matlab which takes the structure zc (see
% FT_zernike_map_convolution), which contains coefficients for different
% Zernike polynomials and adds the additional structure zc.Anm.  The 
% original zc contains the relative amplitudes of the odd and even 
% polynomials.  The new structure zc.Anm combines the amplitudes of the odd
% and even polynomials, for each n,m pair and stores the overall amplitude
% in a structure described by n,m.
%
% zc:       Structure storing odd and even Zernike amplitudes.
%
% zc.Anm:   Structure storing overall amplitudes for polynomials with given 
%           n,m (i.e. combining amplitudes of odd/ even polynomials).
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    21.03.2012
%--------------------------------------------------------------------------
%

function [zc] = FT_reshuffle_zernike_content(zc)

    % Find maximum n
    nmax = max(max(zc.n));
    
    % Initiate new structure zc.Anm
    ps = ceil((nmax+1)/2);
    zc.Anm.amp(1:nmax+1,1:ps) = nan;
    zc.Anm.n(1:nmax+1,1:ps) = nan;
    zc.Anm.m(1:nmax+1,1:ps) = nan;
    
    % For each order (n) polynomial, reshuffle the odd and even amplitudes
    % into one value for each m
    for n=0:nmax
        
        i=n+1;
        j=0;
        
        % If n is even we start m from 0, otherwise from 1
        if n/2==floor(n/2)
            m_start=0;
        else
            m_start=1;
        end
        
        % Loop through all possible values of m for given n
        for m=m_start:2:n
            
            j=j+1;
            
            % If m=0 there is no odd polynomial
            if m==0
                zc.Anm.amp(i,j)=zc.amp(i,n/2+1);
                zc.Anm.n(i,j)=n;
                zc.Anm.m(i,j)=m;
                
            % Otherwise the combined amplitude is the root of the sum of
            % the sqaures of the even and odd amplitudes
            else
                zc.Anm.amp(i,j)=sqrt(zc.amp(i,(n-m)/2+1)^2+zc.amp(i,(n+m)/2+1)^2);
                zc.Anm.n(i,j)=n;
                zc.Anm.m(i,j)=m;
            end 
        end    
    end
        
end

