%--------------------------------------------------------------------------
% function [zc] = FT_list_zernike_content(zc,no_summed,option)
% 
% A function for Matlab which lists the zernike polynomials in the
% structure zc in order of amplitude (i.e. largest first) or in Zernike 
% order (n) order.  It plots the sum of these polynomials.  The list is
% produced in a table in latex syntax.
%
% zc:           Structure containing zernike amplitudes                            
% no_summed:    The number of polynomials to find.  Finds the largest 
%               amplitude (or order) first and then progressivley smaller 
%               ones.
% option:       Indicates the type of list
%               0: lists the polynomials from low to high orders.
%               1: lists the polynomials in decreasing amplitude order.
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    11.01.10
%--------------------------------------------------------------------------
%
function [zc] = FT_list_zernike_content(zc,no_summed,option)
    
    baseid = 'list_zernike_content';

    % Error message
    % option must be 0 or 1
    if (option~=0 && option~=1)
        result='Invalid option: must be 0 or 1';
        msgid=[baseid,':checkarguments']
        error(msgid,result);
    end
    
    % Resyructure zc
    zc = FT_reshuffle_zernike_content(zc);
    tmp_amp = zc.Anm.amp';
    tmp_n = zc.Anm.n';
    tmp_m = zc.Anm.m';
    
    % Find maximum n
    nmax = max(max(zc.n));
    large_amps = false;
    Zmap_data=0;
    
    % Coordinates
    xsize = 500;
    ysize = 500;
    x0 = (xsize+1)/2;
    y0 = (ysize+1)/2;
    rx = (xsize-1)/2;
    ry = (ysize-1)/2;
    
    % Total number of polynomials in zc 
    k = nmax+1;
    N = ceil(k/2)*(ceil(k/2)+1)/2 + floor(k/2)*(floor(k/2)+1)/2;
    
    if no_summed > N
        no_summed = N;
    end
    
    % Table header for results
    disp('    n    &     m    &    Amplitude   \\');
    
    j=1;
    while j<=no_summed
        
        if option == 0      
            [idx] = find(isnan(tmp_amp)==0);
            [loc] = idx(j);
        elseif option == 1
            % Find the largest amplitude polynomial
            [loc]=find(abs(tmp_amp)==max(max(abs(tmp_amp))));
        end
        
        
        if (tmp_amp(loc(1))==0)
            % If the amplitude is zero terminate the loop 
            disp('No more polynomials with amplitude greater than 0');
            disp(sprintf('Number of contributing polynomials: %g',j-1));
            j=no_summed+1;
        else
    
            for i=1:length(loc)
                % Don't plot the very large amplitude zernike polynomials
                % (i.e. curvature) unless the large polynomials are the only
                % things being plotted.
                if (tmp_amp(loc(i))>=1000 && no_summed~=length(loc))
                    
                    n=tmp_n(loc(i));
                    m=tmp_m(loc(i));
                    A=tmp_amp(loc(i));
                
                    % List polynomial and amplitude
                    output=[num2str(n) '   &   ' num2str(m) '   &   ' num2str(A) '   \\'];
                    disp((output));
                
                    large_amps=true;
                else
            
                    n=tmp_n(loc(i));
                    m=tmp_m(loc(i));
                    A=tmp_amp(loc(i));
                
                    % Amplitude of individual odd and even polynomials
                    if m~=0
                        A_odd = FT_get_zernike_coefficient(zc,n,-m);
                    else
                        A_odd = 0;
                    end
                    
                    A_even = FT_get_zernike_coefficient(zc,n,m);
                    
                    % Add the zernike polynomial to the surface to be plotted
                    Zmap_data=Zmap_data+A_odd*FT_zernike_polynomial_for_grid(xsize,ysize,x0,y0,rx,ry,n,-m);
                    Zmap_data=Zmap_data+A_even*FT_zernike_polynomial_for_grid(xsize,ysize,x0,y0,rx,ry,n,m);
                
                    % Add to list
                    output=[num2str(n) '   &   ' num2str(m) '   &   ' num2str(A) '   \\'];
                    disp((output));
                end
    
                % Set the amplitude of the polynomial to 0 so the next largest
                % amplitude polynomial can be found
                tmp_amp(loc(i))=0;
        
                j=j+1;
            end
        end
    end
    
    % Create map from Zernike data
    [rows,cols]=size(Zmap_data);
    x0=(cols+1)/2;
    y0=(rows+1)/2;
    z_map=FT_new_surface_map('Zernike_map',Zmap_data,0,0,x0,y0,1,1,1e-9);
    z_map=FT_recenter_mirror_map(z_map);
    
    if (large_amps==true)
        z_map=FT_remove_offset_from_mirror_map(z_map,1e-2);
    end
    cmap=FT_flip_colormap;
    colormap(cmap);
    
    % Plot the zernike surface
	FT_plot_mirror_map(z_map,0);
    
end

