%--------------------------------------------------------------------------
% FTE_virgo_thermal_distortion.m
%
% A Matlab example which calculates the thermal distortion of a mirror in
% the arm cavities of the virgo detector, for 1W absorption in the mirror 
% coating.  This example illustrates the use of Finesse Tools for 
% calculating thermal distortions and creating mirror maps representing the
% distortions for use in Finesse simulations.
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    14.02.2012
%--------------------------------------------------------------------------
%

clear all;

% Coating absorption
option = 'coating';

% Number of terms to include in the expansion
num_terms = 100;

% Beam parameters (Virgo parameters)
P = 1;
p = 0;
l = 0;
w = 2e-2;
lambda = 1064e-9;
    
% Initiate thermal beam
beam = FT_init_thermal_beam(P,p,l,w,lambda);
    
% Mirror parameters
dn_dT = -0.87e-5;
% For 1W absorption have Cabs = 1 and P = 1
Cabs = 1;
K = 1.38;
E = 0.8;
alpha = 5.4e-7;
Y = 7.3e10;
pois = 0.17;
a = 0.175;
h = 0.1;
% For 1W absorption
Babs = 1/(P*h);
    
% Initiate thermal mirror
mirror = FT_init_thermal_mirror(dn_dT,Cabs,Babs,K,E,alpha,Y,pois,a,h);
    
% Radial coordinates
r = -a:0.001:a;
    
% Overall thermal distortion
[U,c,p] = FT_thermal_distortion(beam,mirror,r,option,num_terms);

% Approximate parabaloid describing the thermal distortion
U_para = c*r.^2 + p;

% Residual distortion (overall - approximate)
U_res = U - U_para;
    
% Create mirror maps representing overall and residual distortions
[Umap,rmap] = FT_mirror_map_from_thermal_distortion(beam,mirror,r,option,num_terms);
    
% Plot results
% Plotting overall thermal distortion and the approximate parabaloid
figure
    hold on;
    plot(r,U,'r','LineWidth',3)
    plot(r,U_para,'b--','LineWidth',3)
    grid on;
    xlabel('r [m] ')
    ylabel('Thermal distortion [m] ')
    title('Thermal distortion')
    legend('Thermal distortion','Approximate paraboloid')
    set(gca,'XLim',[-0.06 0.06])
    set(gca,'YLim',[-1.5e-7 0])
       
% Plotting the residual thermal distortion
figure
    plot(r,U_res,'r','LineWidth',3)
    grid on;
    xlabel('r [m] ')
    ylabel('Thermal distortion [m] ')
    title('Residual thermal distortion ')
    set(gca,'XLim',[-0.06 0.06])

% Plotting thermal distortion
FT_plot_mirror_map(Umap,0)

% Plotting residual distortion
FT_plot_mirror_map(rmap,0)
    
