%--------------------------------------------------------------------------
% function [Z,c,p] = FT_thermal_lens(beam,mirror,r,option,num_terms)
%
% A function for Matlab which calculates the thermal lens in a mirror due 
% to an incident beam and absorption in the coating or substrate (bulk) and
% returns the lens as well as the approximate (Gaussian weighted) 
% paraboloid.
%
% beam:     Structure containing beam parameters (FT_init_thermal_beam.m)
% mirror:   Structure containing mirror parameters 
%           (FT_init_thermal_mirror.m)
% r:        Radial coordinate [m]
% option:   'coating':  Calculate thermal lens from coating absorption.
%           'bulk':     Calculate thermal lens from bulk absorption.
% num_terms:Number of terms to include in approximation of thermal lens
%           (usually 50 is sufficient).
%
% Z:        Returned thermal lens [m]
% c,p:      Returned curvature (c) and piston (p) of approximate paraboloid
%           for the thermal lens:   Z = c*r.^2 + p
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    07.02.2012
%--------------------------------------------------------------------------
%

function [Z,c,p] = FT_thermal_lens(beam,mirror,r,option,num_terms)

    % Calculate thermal lens from coating absorption
    if strcmp(option,'coating')
        
        [Z,coeffs] = FT_thermal_lens_from_coating_absorption(beam,mirror,r,num_terms);
        [c,p] = FT_approximate_paraboloid_for_thermal_lens(coeffs,mirror.X,mirror.a,beam.w);
        
    % thermal lens from bulk absorption
    elseif strcmp(option,'bulk')
       
        [Z,coeffs] = FT_thermal_lens_from_bulk_absorption(beam,mirror,r,num_terms);
        [c,p] = FT_approximate_paraboloid_for_thermal_lens(coeffs,mirror.X,mirror.a,beam.w);
        
    end

end

