%--------------------------------------------------------------------------
% [Temp] =
% FT_temperature_field_from_bulk_absorption(beam,mirror,r,z,num_terms)
%
% A function for Matlab which approximates the temperature field in a
% mirror heated by a beam as a sum of bessel functions, where the heating
% is due to absorption in the bulk (substrate).
%
% N.B. Currently only compatible for an incidnet LG00 mode.
%
% beam:     Structure storing beam parameters (see FT_init_thermal_beam.m)
% mirror:   Structure storing mirror parameters (see
%           FT_init_thermal_mirror.m)
% r:        Radial coordinate [m]
% z:        Depth coordinate along beam axis [m]
% num_terms:Number of terms to use in approximation
%
% Temp:     Returned temperature field
%
% Equations taken from Vinet 2009, Thermal issues in Advanced GW detectors 
% and the Virgo Optics book.
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    10.02.2012
%--------------------------------------------------------------------------
%

function [Temp] = FT_temperature_field_from_bulk_absorption(beam,mirror,r,z,num_terms)

    % Currently only valid for LG00
    p = 0;
    l = 0;

    % Beam and mirror parameters
    P = beam.P;
    w = beam.w;
    Babs = mirror.Babs;
    a = mirror.a;
    h = mirror.h;
    K = mirror.K;
    X = mirror.X;
    
    % Beam is axis symmetric
    r_abs = abs(r);

    % Compute coefficients
    zeta_0s = FT_bessel_zeros(0,X,num_terms);
    p_0s = FT_calculate_fourier_bessel_coefficients(0,0,X,w,a,num_terms);
    gamma_0s = zeta_0s*h/(2*a);
    d_10s = zeta_0s.*sinh(gamma_0s) + X*cosh(gamma_0s);
    
    % Approximating temperature field as bessel sum, including num_terms
    % terms in the approximation
    for j=1:length(z)
        T = 0;
        for s=1:num_terms
            T_term = Babs*P/(pi*K) * p_0s(s)/(zeta_0s(s)^2) * (1 - (X*cosh(zeta_0s(s)*z(j)/a) / d_10s(s)));  
            T = T + T_term*besselj(0,zeta_0s(s)*r_abs/a);
        end
        Temp(j,:) = T(:);
    end
    
end

