%--------------------------------------------------------------------------
% function [lens_map,res_map] =
% FT_phase_map_from_thermal_lens(beam,mirror,r,option,num_terms)
%
% A function for Matlab which creates mirror map structures representing
% the optical path length (or lens) due to the thermal effect of a laser
% beam incident on a mirror.  The function returns a phase map describing
% the overall thermal lens and the residual thermal lens after a Gaussian
% weighted paraboloid lens is fitted and removed from the thermal lens.
% The radius of curvaure of this fitted and removed lens is also returned.
%
% beam:     Structure storing the parameters of the incident beam (see
%           FT_init_thermal_beam.m).
% mirror:   Structure storing the mirror parameters (see
%           FT_init_thermal_mirror.m).
% r:        Vector of radial coordinates [m]
% option:   'coating':  calculates lens from absorption in the coating
%           'bulk':     lens from absorption in the bulk
% num_terms:Number of terms to include in thermal approximation (50 should
%           be sufficient). 
%
% lens_map: Returned phase map of thermal lens in map structure (see 
%           FT_new_surface_map.m).
% res_map:  Residual thermal lens after removal of the approximate
%           paraboloid.
% Rc:       Radius of curvature of approximate thermal lens [m]
%
% Charlotte Bond    27.01.2013
%--------------------------------------------------------------------------
%

function [lens_map,res_map,Rc] = FT_phase_map_from_thermal_lens(beam,mirror,r,option,num_terms)
    
    % Calculate 1D optical path length caused by thermal effects
    [Z,c,p] = FT_thermal_lens(beam,mirror,r,option,num_terms);

    % Setup map parameters
    cols = length(r);
    rows = length(r);
    
    % x-/y-axis
    x = r;
    y = r;
    
    % Into polar coordinates
    [X,Y] = meshgrid(r);
    R = sqrt(X.^2+Y.^2);
    theta = atan2(Y,X);
    
    % Calculate 2D optical path length
    Z_lens = FT_thermal_lens(beam,mirror,R,option,num_terms);
    
    % Approximate paraboloid
    Z_para = c*R.^2+p;
    
    % Set outside the mirror aperture to nan
    for i=1:rows
        for j=1:cols
            if R(i,j)>mirror.a
                Z_lens(i,j)=nan;
                Z_para(i,j)=nan;
            end
            
        end
    end
    
    % Residual data (difference from approximate paraboloid)
    Z_res = Z_lens - Z_para;
    
    % Radius of curvature
    Rc = 1/(2*c);
    
    % Map parameters
    % Phase map
    type = 0;
    
    % Affects transmitted light only
    field = 2;
    
    % Scaling factors and optical centre
    x0 = (cols+1)/2;
    y0 = (rows+1)/2;
    xstep = x(2)-x(1);
    ystep = y(2)-y(1);
    
    % Data in nm
    scaling = 1e-9;
    Z_lens = Z_lens/scaling;
    Z_para = Z_para/scaling;
    Z_res = Z_res/scaling;
    
    % Create thermal lens phase maps
    lens_map = FT_new_surface_map('thermal_lens',Z_lens,type,field,x0,y0,xstep,ystep,scaling);
    res_map = FT_new_surface_map('thermal_lens_residual',Z_res,type,field,x0,y0,xstep,ystep,scaling);
   
    
end

