%--------------------------------------------------------------------------
% function [c,p] =
% FT_approximate_paraboloid_for_thermal_distortion(beam,mirror,num_terms)
%
% A function for Matlab which calculates the approximate paraboloid for the
% thermal distortion of a mirror.
%
% beam:     Structure storing beam parameters (FT_init_thermal_beam.m).
% mirror:   Structure storing mirror parameters (FT_init_thermal_mirror.m).
% num_terms:Number of terms to use in approximation of thermal distortion.
%
% c,p:      Parameters for best fitted paraboloid:
%           Z = c*r^2+p
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    13.02.2012
%--------------------------------------------------------------------------
%

function [c,p] = FT_approximate_paraboloid_for_thermal_distortion(beam,mirror,num_terms)

    % Beam parameters
    P = beam.P;
    w = beam.w;

    % Mirror parameters
    alpha = mirror.alpha;
    pois = mirror.pois;
    Cabs = mirror.Cabs;
    K = mirror.K;
    a = mirror.a;
    h = mirror.h;
    X = mirror.X;
    
    % Factors
    k1 = - alpha*(1+pois)*Cabs*P/(4*pi*a^2*K);
    k2 = - alpha*(1+pois)*Cabs*P/(pi*K);

    % Calculate coefficents
    zeta_0s = FT_bessel_zeros(0,X,num_terms);
    x_0s = zeta_0s.^2 * w^2 / (8*a^2);
    
    % Initial paraboloid parameters
    c = 0;
    p = 0;
    
    % Calculate parabaloid terms
    for s=1:num_terms
        
        c = c + k1 * (zeta_0s(s)^3*exp(-zeta_0s(s)^2*w^2/(4*a^2))/((zeta_0s(s)^2+X^2)*(besselj(0,zeta_0s(s))^2))) * (zeta_0s(s)+X-(zeta_0s(s)-X)*exp(-2*zeta_0s(s)*h/a)) / ((zeta_0s(s)+X)^2-(zeta_0s(s)-X)^2*exp(-2*zeta_0s(s)*h/a));
        p = p + k2 * (zeta_0s(s)*exp(-x_0s(s))/((zeta_0s(s)^2+X^2)*(besselj(0,zeta_0s(s))^2))) * (1-(1+x_0s(s))*exp(-x_0s(s))) * (zeta_0s(s) + X - (zeta_0s(s)-X)*exp(-2*zeta_0s(s)*h/a)) / ((zeta_0s(s)+X)^2-(zeta_0s(s)-X)^2*exp(-2*zeta_0s(s)*h/a));  
        
    end


end

