%--------------------------------------------------------------------------
% function [Lrt] =
% FT_FP_cavity_round_trip_loss(Pin,Ploss,lambda,L,T1,R1,R2)
%
% A function for Matlab which calculates the round-trip loss of a cavity by
% comparing the power in a perfect cavity with the power circulating in a 
% lossy cavity (i.e. from a FINESSE simulation).  N.B. the circulating 
% power should be that at a1 (intra-cavity power reflected from the ITM , 
% i.e. not a3/a3', the power after reflection from the ETM).  Ideal for
% calculating the loss due to mirror surface distortions.
%
% Pin:      Power injected into the cavity [W]
% Ploss:    Power circulating in a lossy cavity [W]
% lambda:   Laser light wavelength [m]
% L:        Cavity length [m]
% T1:       Transmission coefficient (power) of the ITM.
% R1/R2:    Reflection coefficients (power) of the ITM/ETM.
%
% Lrt:      Returned round-trip loss
% P0:       Circulating power in perfect cavity [W]
%
% Charlotte Bond    24/10/2012
%--------------------------------------------------------------------------
%

function [Lrt,P0] = FT_FP_cavity_round_trip_loss(Pin,Ploss,lambda,L,T1,R1,R2)
    
    % Wavenumber
    k = 2*pi/lambda;

    % Amplitude reflection coefficients
    r1 = sqrt(R1);
    r2 = sqrt(R2);

    % Calculate circulating power in perfect cavity
    P0 = Pin * T1/(1+R1*R2-2*r1*r2);

    % Calculate round-trip loss
    Lrt = T1*(sqrt(P0/Ploss)-1);

end

