%--------------------------------------------------------------------------
% function [zmap,Rcx,Rcy] =
% FT_write_astigmatic_map(A,Rc,R,realsize,gridsize)
%
% A function for Matlab which writes a mirror map file in the FINESSE
% format.  The map describes the astigmatic part of a specific mirror 
% surface.  The function also returns the equivalent astigmatism in RCx/Rcy
% values.  Ideal for comparing methods for describing astigmatism in
% FINESSE.
%
% A:        The astigmatism amplitude [nm]
% Rc:       The radius of curvature of the mirror.  This value is only
%           required to calculate the equivalent Rcx/Rcy values. [m]
%           For a flat mirror use Rc=0.
% R:        Mirror radius [m]
% realsize: Real size of the mirror map (i.e. length of one side) [m]
% gridsize: Number of points for the grid representing the map, points =
%           gridsize*gridsize.
%
% zmap:     Map describing astigmatic surface
% Rcx/Rcy:  Equivalent radii of curvature in x/y direction [m]
%
% Charlotte Bond    17.01.2013
%--------------------------------------------------------------------------
%

function [zmap,Rcx,Rcy] = FT_write_astigmatic_map(A,Rc,R,realsize,gridsize,filename)
    
    % Create astigmatic map
    zmap = FT_create_individual_Znm_map(2,2,A,R,realsize,gridsize,0,0);
    
    % Calculate Z20 amplitude, representing common curvature (in nm)
    A20 = (1/2)*(Rc-sign(Rc)*sqrt(Rc^2-R^2));
    A20 = A20*1e9;
    
    % Flat mirror if Rc=0
    if Rc==0
        A20=0;
    end
    
    % Vector of zernike amplitudes and indices
    Amps = [0,A20,A];
    n = [2,2,2];
    m = [-2,0,2];
    
    % Calculate equivalent curvatures
    OF = FT_zernike_optical_features(Amps,R,n,m);
    Rcx = OF.value(2);
    Rcy = OF.value(1);
    
    % Output equivalent curvatures
    fprintf('Rcx:     %g \n',Rcx)
    fprintf('Rcy:     %g \n',Rcy)
    
    % Write map to file
    FT_write_surface_map(filename,zmap);
    
end

