%--------------------------------------------------------------------------
% function [map_out,Rc_out] = FT_remove_zernike_curvatures_from_map(map,curvature)
%
% A Matlab function that preforms an inner product between a map surface 
% and particular Zernike polynomials, removing the resulting polynomials
% from the map surface.  The polynomials used here are the Zernike 
% polynomials defining mirror curvature, Z02, constant curvature, and
% Z-22 and Z+22, astigmatism.
%
% map:          The map surface from which the curvatures will be removed.
% curvature:    'all':          Astigmatism and constant curvature
%               'constant':     Just remove constant curvature (i.e. Z_0^2)
%               'astigmatism':  Just remove astigmatism
%
% map_out:      Returned map with curvatures removed
% Rc_out:       Equivalent curvature removed.  Set to 0 if just astigmatism
%               is being removed. 
% A:            Amplitudes of removed polynomials
% 
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    09.02.11
%--------------------------------------------------------------------------
%

function [map_out,zc]=FT_remove_zernike_curvatures_from_map(map,curvature)
    
    % Error messages
    if (~strcmp(curvature,'astigmatism') && ~strcmp(curvature,'constant')...
             && ~strcmp(curvature,'all'))
        msg = 'Invalid value: curvature must be "constant", "astigmatism" or "all"'
        msgid = [baseid,':checkarguments']
        error(msgid,result);
    end

    % Zernike map convolution up to order 2 (don't save the data)
    zc = FT_zernike_map_convolution(map,2,0);
   
    % All curvatures have n = 2
    n=2;
  
    % Set values of m for different conditions
    if (strcmp(curvature,'constant'))
        mc=0;
    elseif (strcmp(curvature,'astigmatism'))
        mc=[-2 2];
    elseif (strcmp(curvature,'all'))
        mc=[-2 0 2];
    end
        
    map_out = map;
    
    % Remove curvature polynomials from the map
    for i=1:length(mc)

        m=mc(i);
      
        A = FT_get_zernike_coefficient(zc,n,m); 
        
        % Remove polynomial
        map_out=FT_remove_zernike_polynomial_from_map(map_out,n,m,A);
    
    end
    
end