%--------------------------------------------------------------------------
% function [z_min,z_max] = FT_plot_surface_profile(profile)
%
% A function for Matlab which plots the surface defined in the profile
% structure (see FT_new_surface_profile.m).
%
% profile:      Structure describing a 1D surface (i.e. mirror surface)
% R:            Radius over which to plot the surface [m].  For the whole
%               profile use R=0.
%
% z_max,z_min:  Maximum and minimum surface heights in range given by R 
%               [nm]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    12.05.2012
%--------------------------------------------------------------------------
%

function [z_max,z_min] = FT_plot_surface_profile(profile,R)
    
    points = numel(profile.data);
    
    % r scale in units of cm
    idx = linspace(1,points,points);
    r = (idx - profile.r0) * profile.rstep * 100;

    % Put R into cm
    R = 100*R;
    
    % Put data into nm
    data = profile.data * profile.scaling * 1e9; 

    % Plot the surface profile
    plot(r,data,'b','LineWidth',2)
    
    % Axis labels
    xlabel('r [cm] ')
    ylabel('Surface height [nm] ')
    
    % Set r limits depending on R
    if R==0
        % r limits
        r_min = min(r);
        r_max = max(r);
        % r limit indices
        i_min = 1;
        i_max = length(r);
    else
        % r limits
        r_min = -R;
        r_max = R;
        % r limit indices
        i1 = find(r>=r_min);
        i_min = min(i1); 
        i2 = find(r<=r_max);
        i_max = max(i2);
    end
    set(gca,'XLim',[r_min r_max])    
    
    grid on;  
    
    % Find maximum and minimum points within data range
    z_max = max(data(i_min:i_max));
    z_min = min(data(i_min:i_max));
    
    
    
    
end

