%--------------------------------------------------------------------------
% function [Rc_out] = FT_combine_curvatures(Rc1,Rc2,R)
%
% A function for Matlab which returns the resulting radius of curvature
% resulting from the sum of two surfaces defined by two different Rcs.  The
% resulting curvature is calculated assuming the curved surfaces are
% described by the Zernike polynomial Z20.  This approximation is valid for 
% R << Rc, and fails for extreme curvatures, i.e. when Rc = R.
%
% Rc1:      Radius of curvature of the first surface.
% Rc2:      Radius of curvature of the second surface.
% R:        Radius of the surface, i.e. radius of the mirror aperture.
%
% Rc_out:   Returned radius of curvature of the combined surfaces 1 and 2.
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    20.02.2012
%--------------------------------------------------------------------------
% 

function [Rc_out] = FT_combine_curvatures(Rc1,Rc2,R)

    baseid = 'combine_curvatures';
    
    % Error messages
    % If Rc1 or Rc2 are smaller than R some parts of the surface may be
    % undefined
    if (Rc1<=R || Rc2<=R)
        result='Invalid radius of curvature: Rc1 and Rc2 must be larger than R';
        msgid=[baseid,':checkarguments']
        error(msgid,result);
    end

    % Amplitude of Z20 defined by the sum of the two surfaces defined by
    % Rc1 and Rc2
    A = (1/2)*(Rc1+Rc2-sign(Rc1)*sqrt(Rc1^2-R^2)-sign(Rc2)*sqrt(Rc2^2-R^2));

    % Equivalent radius of curvature 
    Rc_out = (4*A^2+R^2)/(4*A);

end

