%--------------------------------------------------------------------------
% function [phi0,phase0,gain] =
% FT_finesse_lock_parameters(FT,katname,nodename,mirrorname,fmod,steps,type,verbose)
%
% A function for Matlab which uses the simulation Finesse to calculate
% several parameters required to lock a cavity in Finesse, where the error 
% signal is produced by a photodiode (PD) in reflection or transmission.
% The parameters include the approximate operating point, the optimum 
% demodulation phase of the PD and the best gain for the lock.
%
% FT:           Structure containing the name and path of the Finesse
%               executable
% katname:      Name of the kat (Finesse) file which contains the structure 
%               of the interferometer to be simulated.  The file should 
%               contain only the optical components, with no detectors or 
%               axes.
% nodename:     Name of the node at which the PD is placed to detect the
%               error signal. (A '*' added to the nodename changes the
%               direction the PD 'looks').
% mirrorname:   Name of the mirror to tune to lock the cavity.
% fmod:         Modulation frequency of the sidebands used to produce the
%               error signal.  The sidebands should be added to the laser
%               light in katname using the mod command [Hz].
% steps:        Integer vector giving 'step' values for the different 
%               x_axis commands needed.
%               steps(1):   Steps required for the cavity scan to 
%                           approximate the operating point.  The scan goes
%                           from 0 to 360 degrees.
%               steps(2):   Steps required to find maximum demodulation
%                           phase, going from 0 to 180 degrees.
%               steps(3):   Steps required to find the slope at the 
%                           operating point.  The mirror is tuned 0.5 
%                           degrees either side of the operating point.
% type:         Type of error signal.  There are only two possible types:
%               'reflection'
%               'transmission'
%               which refer to an error signal being taken in transmission
%               or reflection.
% verbose:      0:  don't plot output
%               1:  plot output
%
% phi0:         Approximate operating point [degrees]
% phase0:       Optimum demodulation phase [degrees]
% gain:         Best gain for lock [deg/W]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    10.10.2012
%--------------------------------------------------------------------------
%

function [phi0,phase0,gain] = FT_finesse_lock_parameters(FT,katname,nodename,mirrorname,fmod,steps,type,verbose)

    baseid = 'finesse_lock_parameters';
    
    % Error messages
    % Only possible error signal types are 'transmission' and 'reflection'
    if (~strcmp(type,'transmission') && ~strcmp(type,'reflection'))
        result='Invalid type: should be "transmission" or "reflection"';
        msgid=[baseid,':checkarguments']
        error(msgid,result);
    end

    % 1st step: We perform a cavity scan to find the approximate
    % opperating point
    x_axis = [0 360 steps(1)];
    scan = FT_finesse_cavity_scan(FT,katname,nodename,mirrorname,x_axis,verbose);
    
    % In transmission we get maximum power at the operating point, in
    % reflection we get minimum power
    if strcmp(type,'transmission')
        [p0,idx] = max(scan.power);
    else
        [p0,idx] = min(scan.power);
    end
    
    % Approximate operating point
    phi0 = scan.tuning(idx(1));    
    
    % 2nd step: Find the optimum phase, i.e. the phase at which the pd
    % response is at a maximum, to use in the lock
    x_axis = [0 180 steps(2)];
    PD = FT_finesse_PD_demodulation_phase(FT,katname,nodename,mirrorname,fmod,phi0,x_axis,verbose);
    
    % Optimum demodulation phase
    [Pmax,idx] = max(PD.signal);
    phase0 = PD.phase(idx(1));
    
    % 3rd step: Find the required gain for the lock.  The slope of the
    % error signal is computed.
    x_axis = [-0.5+phi0 0.5+phi0 steps(3)];
    err_sig = FT_finesse_error_signal_slope(FT,katname,nodename,mirrorname,fmod,phi0,phase0,x_axis,verbose);
    
    % The best gain for the lock is minus the inverse slope at the
    % operating point
    % Find slope at phi0
    [idx] = find(err_sig.tuning>=phi0);
    slope0 = err_sig.slope(idx(1));
    
    % Calculate gain
    gain = -1/slope0;

end

