%--------------------------------------------------------------------------
% function [err_sig] =
% FT_finesse_error_signal_slope(FT,katname,nodename,mirrorname,fmod,phi0,phase0,x_axis,verbose)
%
% A function for Matlab which runs the interferometer simulation Finesse to
% find the slope of a photodiode (PD) error signal for a given setup.  
% This function can be used to compute the required gain for a cavity lock 
% (see FT_finesse_lock_parameters.m) at the operating point.
%
% FT:           Structure containg name and path of Finesse executable
% katname:      Name of the kat (Finesse) file which contains the structure
%               of the interferometer to be simulated.  The file should
%               contain only optical components, with no detectors or
%               x/yaxis commands.
% nodename:     Name of the node at which to place the PD. (Adding '*' to
%               the nodename changes the direction the PD 'looks').
% mirrorname:   Name of the cavity mirror to tune, around the operating
%               point, to see the response of the error signal slope.
% fmod:         Modulation frequency of the sidebands added to the laser
%               light in katname, by the mod command [Hz].
% phi0:         Approximate operating point [degrees].
% phase0:       Optimum demodulation phase of the PD at phi0 [degrees].
% x_axis:       x-axis parameters describing mirror tuning
%               x_axis(1):  minimum tuning [degs]
%               x_axis(2):  maximum tuning [degs]
%               x_axis(3):  x-axis steps
% verbose:      0: don't plot output
%               1: plot output
%
% err_sig:      Structure storing simulation results
%               err_sig.tuning:     Mirror tuning [degrees]
%               err_sig.slope:      Slope of the PD error signal [W/deg]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    10.10.2012
%--------------------------------------------------------------------------
%

function [err_sig] = FT_finesse_error_signal_slope(FT,katname,nodename,mirrorname,fmod,phi0,phase0,x_axis,verbose)

    % x-axis parameter
    xmin = x_axis(1);
    xmax = x_axis(2);
    steps = x_axis(3);

    % Copy blocks from input file
    outbl = FT_read_blocks_from_file(katname);
    nb = length(outbl);
    
    % Create new block for error signal simulation
    outbl(nb+1) = FT_create_new_block('error_signal_slope');

    % Photodetector at given node, with given phase and demodulating at the
    % given sideband frequency
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('pd1 pd_error %g %g %s',fmod,phase0,nodename));

    % Differential of the error signal
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('diff %s phi',mirrorname));
    
    % Axes
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('xaxis %s phi lin %g %g %g',mirrorname,xmin,xmax,steps));
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('yaxis abs'));

    % New katname
    katname = FT_remove_file_extension(katname);
    katname = sprintf('%s_error_signal_slope.kat',katname);
    
    % Write new katfile
    FT_write_blocks_into_file(outbl,katname);
    
    % Run finesse and read data
    Run = FT_create_new_kat_run();
    Run.filename = katname;
    Run = FT_run_kat_simulation(FT,Run,1);
    
    % Store data
    err_sig.tuning = Run.x;
    err_sig.slope = Run.y;
    
    if verbose==1
        % Plot error signal slope vs. tuning
        figure
            plot(err_sig.tuning,err_sig.slope);
            title('Error signal slope vs. mirror tuning')
            xlabel('tuning [degrees] ')
            ylabel('error signal slope [W/deg] ')
            drawnow;
            
        % Find slope at phi0
        [idx] = find(err_sig.tuning>=phi0);
        slope0 = err_sig.slope(idx(1));
            
        fprintf(' -> Slope at the given operating point :   %g [W/deg] \n',slope0);
    end


end

