%--------------------------------------------------------------------------
% function [error] =
% FT_finesse_error_signal(FT,katname,nodename,mirrorname,fmod,phase0,x_axis,verbose)
%
% A Matlab function which runs the interferometer simulation Finesse and
% calculates the error signal (from a photodiode, PD) for a given 
% interferometer setup.  The error signal is computed as a function of the 
% tuning of a mirror in the setup and requires sidebands added to the input
% laser beam.
%
% FT:           Structure storing the name and location of the Finesse
%               executable.
% katname:      Finesse (kat) file describing the interferometer setup,
%               without detectors or axis commands.
% nodename:     Name of the node at which to place the photodiode.
% mirrorname:   Name of the mirror to tune.
% fmod:         Modulation frequency of the input beam sidebands [Hz].
% phase0:       Demodulation phase for the PD (for the optimum phase see 
%               FT_finesse_lock_parameters.m) [deg]
% x_axis:       Variables defining the x-axis range, i.e. the tuning of the
%               given mirror. 
%               x_axis(1):      initial mirror tuning [deg]
%               x_axis(2):      final mirror tuning [deg]
%               x_axis(3):      number of tuning steps
% verbose:      0:  don't plot output
%               1:  plot output
%
% error:        Returned error signal
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    15.10.2012
%--------------------------------------------------------------------------
%

function [error] = FT_finesse_error_signal(FT,katname,nodename,mirrorname,fmod,phase0,x_axis,verbose)

    % Mirror tuning parameters
    xmin = x_axis(1);
    xmax = x_axis(2);
    steps = x_axis(3);

    % Copy blocks from the input file
    outbl = FT_read_blocks_from_file(katname);
    
    % Number of blocks in katname
    nb = length(outbl);
    
    % Create new block storing commands for the error signal
    outbl(nb+1) = FT_create_new_block('error_signal');
    
    % Add photodetector
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('pd1 pd_error %g %g %s',fmod,phase0,nodename));
    
    % Tune mirror position along x-axis
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('xaxis %s phi lin %g %g %16.0f',mirrorname,xmin,xmax,steps));
    
    % y-axis
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),'yaxis abs');
    
    % New katname
    katname = FT_remove_file_extension(katname);
    katname = sprintf('%s_error_signal.kat',katname);
    
    % Write katfile
    FT_write_blocks_into_file(outbl,katname);

    fprintf('Error signal from %g [degrees] to %g [degrees] \n',xmin,xmax)
    % Run finesse and read the simulation data
    Run = FT_create_new_kat_run();
    Run.filename = katname;
    Run = FT_run_kat_simulation(FT,Run,1);
    
    % Store output
    error.tuning = Run.x;
    error.signal = Run.y;
    
    if verbose == 1
        % Plot error signal
        figure
            plot(error.tuning,error.signal);
            title('Error signal')
            xlabel('tuning [degrees]')
            ylabel('abs')
            drawnow;
            
    end


end

