%--------------------------------------------------------------------------
% function [scan] =
% FT_finesse_cavity_scan(FT,katname,nodename,mirrorname,xmin,xmax,steps,verbose)
%
% A function for Matlab which writes and performs a cavity scan in the
% interferometer simulation Finesse, i.e. tuning a mirror of a sepecified 
% cavity and detecting the power at a given point.  The function requires 
% an input kat file which defines the Finesse structure of the cavity/ 
% interferometer, the name of the node at which to detect the power and the
% name of the mirror to tune.
%
% FT:           Structure containing path and name of Finesse executable
% katname:      Name of the kat (Finesses) file which details the structure 
%               of the interferometer to be simulated, with no detection/ 
%               xaxis commands.  The function writes a new katfile with the
%               cavity scan commands.
% nodename:     Name of the node in the input file at which to detect the 
%               power during the cavity scan.  (Note that to change the
%               direction the photodetector 'looks' add a '*' to the node 
%               name.) 
% mirrorname:   Name of the mirror to tune in the scan
% x_axis:       x-axis parameters describing the mirror tuning
%               x_axis(1):  minimum tuning [degs]
%               x_axis(2):  maximum tuning [degs]
%               x_axis(3):  number of x-axis steps
% verbose:      0: don't plot output
%               1: plot output
%
% scan:         Structure storing the results of the scan
%               scan.tuning:    Vector of tunings of the mirror [degrees]
%               scan.power:     Power at the given node over the tunings [W]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    08.08.2012
%--------------------------------------------------------------------------
%

function [scan] = FT_finesse_cavity_scan(FT,katname,nodename,mirrorname,x_axis,verbose)

    % x-axis parameters
    xmin = x_axis(1);
    xmax = x_axis(2);
    steps = x_axis(3);

    % Copy blocks from the input file
    outbl = FT_read_blocks_from_file(katname);

    % Number of blocks in katname
    nb = length(outbl);
    
    % Create new block storing the commands for the cavity scan
    outbl(nb+1) = FT_create_new_block('cavity_scan');
    
    % Add photodetector to detect the power at the given node  
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('pd pd_scan %s',nodename));
    
    % Add x-axis line, defining the mirror to be tuned and the range and
    % number of tunings
    xline = sprintf('xaxis %s phi lin %g %g %16.0f',mirrorname,xmin,xmax,steps);
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),xline);
    
    % y-axis
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),'yaxis abs');
    
    % New katname
    katname = FT_remove_file_extension(katname); 
    katname = sprintf('%s_cavity_scan.kat',katname);
    
    % Write katfile
    FT_write_blocks_into_file(outbl,katname);
    
    fprintf('Scanning cavity from %g [dgerees] to %g [degrees] in %g steps \n',xmin,xmax,steps)
    % Run Finese and read the simulation data
    Run = FT_create_new_kat_run();
    Run.filename = katname;
    Run = FT_run_kat_simulation(FT,Run,1);
    
    % Store output data
    scan.tuning = Run.x;
    scan.power = Run.y;
    
    if verbose==1
        % Plot cavity scan
        figure
            semilogy(scan.tuning,scan.power);
            title('Cavity scan')
            xlabel('tuning [degrees]')
            ylabel('power [W]')
            drawnow;
        
        % Find the maximum power and relevant tuning
        Pmax = max(scan.power);
        idx = find(scan.power==Pmax);
        OP = scan.tuning(idx(1));
        
        fprintf(' -> Max power of %f [W] at %f [degrees] \n',Pmax,OP); 
    end
    
end

