%--------------------------------------------------------------------------
% function [beam] =
% FT_finesse_beam_detection(FT,katname,nodename,mirrorname,phi0,gridsize,gridsteps,verbose)
%
% A function for Matlab which writes and runs a Finesse simulation to
% detect the beam circulating in a given interferometer setup at a given
% point.  The function requires an input kat file which defines the
% interferometer structure, the name of the node at which to detect the
% beam and the name of the mirror to tune (i.e. for cavities, when the beam
% is detected at a given operating point).
%
% FT:           Structure containing path and name of Finesse executable
% katname:      Original kat file, which details the structure of the
%               interferometer to be simulated.  A new kat file is written,
%               containing the initial setup and beam detection commands.
% nodename:     Name of the node at which to detect the beam.  (To look in
%               the other direction add '*' to the nodename.)
% mirrorname:   Mirror to tune before beam detection
% phi0:         Tuning of given mirror
% gridsize:     Dimensions of the x and y axis (in units of w0).  x/y-axis
%               goes from -(gridsize/2)*w0 to (gridszie/2)*w0.
% gridsteps:    Number of steps for x/y-axis
% verbose:      0: no output plot
%               1: plot output
%
% beam:         Returned structure storing results of the simulation
%               beam.xaxis:     Vector of x values [w0]
%               beam.yaxis:     Vector of y values [w0]
%               beam.field:     Complex values representing detected field
%                               [sqrt(W)]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    09.08.2012
%--------------------------------------------------------------------------
%

function [beam] = FT_finesse_beam_detection(FT,katname,nodename,mirrorname,phi0,gridsize,gridsteps,verbose)

    % Copy blocks from the input file
    outbl = FT_read_blocks_from_file(katname);
    
    % Number of blocks in katname
    nb = length(outbl);
    
    % Create new block storing the commands to detect the beam
    outbl(nb+1) = FT_create_new_block('beam_detection');
    
    % Add beam detector at the given node, detecting at carrier frrequency
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('beam beam0 0 %s',nodename));
    
    % Tune given mirror
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('func tuning = %16.8f',phi0));
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('put %s phi $tuning',mirrorname));
    
    % Add lines defining x and y axis
    xline = sprintf('xaxis beam0 x lin %g %g %g',-gridsize/2,gridsize/2,gridsteps);
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),xline);
    yline = sprintf('x2axis beam0 y lin %g %g %g',-gridsize/2,gridsize/2,gridsteps);
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),yline);

    % Add line defining y axis
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),'yaxis re:im');
    
    % New katname
    katname = FT_remove_file_extension(katname);
    katname = sprintf('%s_beam_detection.kat',katname);
    
    % Write katfile
    FT_write_blocks_into_file(outbl,katname);
    
    fprintf('Detecting beam at node %s, at %s tuning %g [degrees] \n',nodename,mirrorname,phi0)
    % Write over original file (just adding commands for beam detection)
    Run = FT_create_new_kat_run();
    Run.filename = katname;
    Run = FT_run_kat_simulation(FT,Run,1);
    
    % Store simulation results
    beam.xaxis = Run.x;
    beam.yaxis = Run.y;
    beam.field = Run.z(:,:,1) + 1i*Run.z(:,:,2);
    
    if verbose==1
        % Plot results
        figure
            FT_FFT_plot_field(beam.field,1)
            title('Detected field')
            drawnow;
    end

end

