%--------------------------------------------------------------------------
% function [PD] =
% FT_finesse_PD_demodulation_phase(FT,katname,nodename,mirrorname,fmod,phi0,x_axis,verbose)
%
% A function for Matlab which runs the simulation Finesse to detect the 
% signal on a photodiode (PD) as a function of the demodulation phase of 
% the PD, for a given Finesse setup.  This function can be used to 
% determine the optimum demodulation phase for use in locking a cavity (see
% FT_finesse_lock_parameters.m).
%
% FT:           Structure containing the path and name of the Finesse
%               executable.
% katname:      Name of the kat (Finesse) file which contains the structure 
%               of the interferometer to be simulated.  The kat file should
%               contain only the optics commponents, with no detectors or
%               x/yaxis commands.
% nodename:     The node in the kat file at which the PD should be placed.
%               (Adding '*' to the nodename changes the direction the PD
%               'looks'.)
% mirrorname:   Name of the mirror to tune to the cavities approximate
%               operating point and to 'shake' to simulate a tranfer
%               function.
% fmod:         Modulation frequency of the sidebands of the eom (mod
%               command) added to the laser light in katname [Hz].
% phi0:         Approximate opperating point of the cavity [degrees].
% x_axis:       Parameters for tuning the demodulation phase on the x-axis
%               x_axis(1):  minimum demodulation phase [degs]
%               x_axis(2):  maximum demodulation phase [degs]
%               x_axis(3):  x-axis steps
% verbose:      0: don't plot output
%               1: plot output
%
% PD:           Structure storing simulation results
%               PD.phase:   Demodulation phase of the PD [degrees]
%               PD.signal:  PD signal [W]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    10.10.2012
%--------------------------------------------------------------------------
%

function [PD] = FT_finesse_PD_demodulation_phase(FT,katname,nodename,mirrorname,fmod,phi0,x_axis,verbose)

    % Axis parameters
    xmin = x_axis(1);
    xmax = x_axis(2);
    steps = x_axis(3);

    % Copy blocks from input file
    outbl = FT_read_blocks_from_file(katname);
    nb = length(outbl);
    
    % Create new block for phase optimisation
    outbl(nb+1) = FT_create_new_block('demodulation_phase');
    
    % Tune mirror to phi0
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('func tuning = %g',phi0));
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('put %s phi $tuning',mirrorname));

    % Shake mirror at 1 Hz for transfer function
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('fsig sig1 %s 1 0',mirrorname));
    
    % Add photodetector at given node with two demodulation frequencies (1
    % Hz and the sideband frequency)
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('pd2 pd_phase %g 0 1 %s',fmod,nodename));
    
    % Axes
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),sprintf('xaxis pd_phase phi1 lin %g %g %g',xmin,xmax,steps));
    outbl(nb+1) = FT_add_line_to_block(outbl(nb+1),'yaxis abs');
    
    % New katname
    katname = FT_remove_file_extension(katname);
    katname = sprintf('%s_demodulation_phase.kat',katname);
    
    % Write new katfile
    FT_write_blocks_into_file(outbl,katname);
    
    % Run finesse and read data
    Run = FT_create_new_kat_run();
    Run.filename = katname;
    Run = FT_run_kat_simulation(FT,Run,1);
    
    % Store data
    PD.phase = Run.x;
    PD.signal = Run.y(:,1);
    
    if verbose==1
        % Plot signal vs. demodulation phase
        figure
            plot(PD.phase,PD.signal);
            title('PD signal vs. demodulation phase')
            xlabel('demodulation phase [degrees] ')
            ylabel('PD signal [W] ')
            drawnow;
            
        % Find optimum phase
        [Pmax,idx] = max(PD.signal);
        phase0 = PD.phase(idx(1));
            
        fprintf(' -> optimum photodiode phase:   %g [degrees] \n',phase0);
    end

end

