%--------------------------------------------------------------------------
% function [result] =
% FT_detect_fields_at_scan_peaks(FT,katname,nodename,mirrorname,x_axis,gridaxis,verbose)
%
% A function for Matlab which runs the interferometer simulation Finesse
% for a given setup.  This function runs a simulation that detects the 
% power at a certain point in the interferometer (a given node), as a
% mirror in the setup is tuned.  The simulation then detects the field at 
% this point at the tunings at which peaks in the power are found.
%
% FT:           Structure containing the path and name of the Finesse
%               executable
% katname:      Name of the Finesse file defining the interferometer
%               structure, without any detectors or axis commands
% nodename:     Name of the node at which to detect the power and the beam
% mirrorname:   Name of the mirror to tune for the scan and beam detection
% x_axis:       Tuning variables for the x-axis for the cavity scan.
%               x_axis(1):  minimum mirror tuning [degs]
%               x_axis(2):  maximum mirror tuning [degs]
%               x_axis(3):  steps for the x-axis
% gridaxis:     Grid variables for the x/y-axis for the beam detection.
%               gridaxis(1):    Range of the x/y-axis [in units of w0].
%                               x/y-axis goes from -gridvar(1)/2 to 
%                               gridvar(1)/2.
%               gridaxis(2):    Number of stesp on the x/y-axis
% verbose:      0:  Cavity scans and beams arn't plotted
%               1:  Scans and beams are plotted
%
% result:       Stores the results of the cavity scan and beam detections
%               result.scan.tuning:     Vector of tunings for the cavity
%                                       scan [degs]
%               result.scan.power:      Vector of power values for the
%                                       cavity scan [W]
%               result.phi0:            Tuning values at peak powers [deg]
%               result.beam(i).xaxis:   x-axis values for the detected 
%                                       field [w0]
%               result.beam(i).yaxis:   Y-axis values for the field [w0]
%               result.beam(i).field:   Field detected at peak powers
%                                       [sqrt(W)]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    11.08.2012
%--------------------------------------------------------------------------
%

function [result] = FT_detect_fields_at_scan_peaks(FT,katname,nodename,mirrorname,x_axis,gridaxis,verbose)

    % Perform cavity scan
    result.scan = FT_finesse_cavity_scan(FT,katname,nodename,mirrorname,x_axis,verbose);
    
    % Find peaks in scan data
    [pks,locs] = findpeaks(result.scan.power);
    
    fprintf('%g peaks located ... \n',length(pks))
    fprintf('... detecting fields \n')
    % Detect the field at each peak tuning
    for i=1:length(pks)
        % Store tuning value
        phi0 = result.scan.tuning(locs(i));
        result.phi0(i) = phi0;
        % Store beam detected at phi0
        result.beam(i) = FT_finesse_beam_detection(FT,katname,nodename,mirrorname,phi0,gridaxis(1),gridaxis(2),verbose);
        
    end
    
end

