%--------------------------------------------------------------------------
% FTE_Winkler_scattering.m
%
% An example script used to plot the coupling into different modes from
% scattering from different spatial wavelengths, as calculated in Winkler's
% approximation.
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    13.05.2012
%--------------------------------------------------------------------------
%

clear all;

plot_name = 'winkler_coupling';

% Beam parameters
w0 = 6.2e-2;            % Beam size at ETM for aLIGO
lambda = 1064e-9;

% Distortion parameters.  Coupling is calculated for a range of different
% spatial wavelengths but a constant amplitude of a nm with no tilt.
A = 1e-9;
s = linspace(0.1,5,1000);
slambda = s*w0;
tilt = 0;
% Incient mode
n = 0;

% Coupled modes
m=0:100;
% Calculate the coupling for different values of m
for i=1:length(m)
    % For odd m-n we need antisymmetric distortions, otherwise symmetric
    % distortions
    dn = abs(m(i)-n);
    if 2*round(dn/2)~=dn
        phi = 90;
    else
        phi = 0;
    end
    
    c(i,:) = FT_Winkler_coupling(lambda,w0,n,m(i),tilt,A,slambda,phi);
    
end

% Calculate power and put into ppm
Ps = 1e6*abs(c).^2;

% Interpolate (just for smoother plot)
Ps = interp2(Ps,2);
[rows,cols] = size(Ps);
s=linspace(min(s),max(s),cols);
m=linspace(min(m),max(m),rows);

% Plot result 
figure
    surfc(s,m,Ps,'edgecolor','none')
    view(2)
    axis tight;
    
    % Positive colour map
    cmap = FT_pos_colormap();
    colormap(cmap);
    
    % Colour bar
    zmax = max(max(Ps));
    h=colorbar;
    
    % Axis labels
    xlabel('Spatial wavelength [w] ')
    ylabel('Mode number, n ')
    ylabel(h,'Power coupled [ppm] ')
    
    % Limits
    set(gca,'XLim',[min(s) max(s)+0.001])
    set(gca,'YLim',[min(m) max(m)])

    grid off;





