%--------------------------------------------------------------------------
% function [Gsag] = FT_sagnac_response(fsig,lambda,L,pc,pr,tc,Amin)
%
% A function for Matlab which computes the transfer function of a Sagnac
% interferometer with specific parameters to a gravitational wave.  The 
% transfer function can be calculated for an interferometer with or without
% Fabry-Perot arm cavities.  The equations in this function are taken from 
% Mizuno 1997.
%
% fsig:     Frequency of the gravitational wave signal [Hz]
% lambda:   Wavelength of the carrier light [m]
% L:        Length of the interferometer arms or arm cavities [m]
% pc:       Amplitude reflectivity of the couling mirror (input mirror)
% pr:       Amplitude reflectivity of the reflective mirror (end mirror)
% tc:       Amplitude transmission of the coupling mirror (input mirror)
%           (tc=1 for case with no FP cavities in the arms).
% Amin:     Loss per reflection in the interferometer
%
% Gsag:     Returned transfer function
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    18.10.2012
%--------------------------------------------------------------------------
%

function [Gsag] = FT_sagnac_response(fsig,lambda,L,pc,pr,tc,Amin)

    % Light parameters
    c=299792458;
    f = c/lambda;
    w0 = 2*pi*f;
    
    % GW signal parameters
    w = 2*pi*fsig;

    % Variables
    T = 2*L/c;
    phi = w*T;
    s = 1i*w;
    
    
    if tc==1
        % Case for no arm cavities
        parm = pr;
        Gsag = parm^2 * w0/4 * (1-exp(-s*T)).^2 ./ s;
    
    else
        % Case for arm cavities
        % Calculating effective reflection of FP cavity
        pcav0 = (pc - (1 - Amin)*pr)/(1-pc*pr);
        pcav = (pc - (1 - Amin)*pr*exp(-1i*phi))./(1 - pc*pr*exp(-i*phi));

        % Response of FP cavities
        Gcav = (tc/(1-pc*pr)) * (-tc./(1-pc*pr*exp(-s*T))) * pr * w0/4 .* (1-exp(-s*T))./s;
        
        % Sagnac response
        Gsag = Gcav .* (pcav0 - pcav);
    end



end

