%--------------------------------------------------------------------------
% function [Gmich] = FT_michelson_response(fsig,lambda,L,pc,pr,tc)
%
% A function for Matlab which calculates the frequency response, or
% transfer function, of a Michelson interferometer to a gravitationl wave
% signal.  The response can be calculated for a Michelson with or without
% Fabry-Perot arm cavities
%
% fsig:     Vector of signal frequencies over which to calculate the 
%           frequency response [Hz]
% lambda:   Wavelength of the lasre light [m]
% L:        Arm length [m]
% pc:       Reflectivity (amplitude) of the coupling mirror (ITM)
%           (for case with no FP cvities pc = 0)
% pr:       Reflectivity (amplitude) of the reflection mirror (ETM)
% tc:       Transmission (amplitude) of the coupling mirror (ITM)
%           (for case with no FP cavities tc = 1)
%
% Gmich:    Returned frequency response for signal frequencies fsig
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond    22.10.2012
%--------------------------------------------------------------------------
%

function [Gmich] = FT_michelson_response(fsig,lambda,L,pc,pr,tc)

    % Light parameters
    c = 299792458;
    f = c/lambda;
    w0 = 2*pi*f;
    
    % Signal parameters
    w = 2*pi*fsig;
    
    % Arm travel time
    T = 2*L/c;
    
    % Fabry-Perot cavity response
    GFP = -tc^2./((1-pc*pr)*(1-pc*pr*exp(-1i*w*T)));

    % Michelson response
    Gmich = GFP * pr*w0/4 .* (1-exp(-1i*w*T))./(1i*w);

end

