
%clear all;

function FTE_ET_optical_parameters()
fprintf('-----------------------------------------------------------------------\n');
fprintf('Example computation of the Einstein Telescope optical design parameters\n');
fprintf('Andreas Freise, 24.04.2011\n\n');
fprintf('At the time of writing this script is consistant with the numbers given\n');
fprintf('in the Einstein Telescope Design Study document, however, the design might\n');
fprintf('have progressed since.\n');
fprintf('-----------------------------------------------------------------------\n');

ftest=303;
Ltest=310;

fprintf('HF detector, 1064nm, LG33\n');
print_parameters(1064e-9,ftest,Ltest);
fprintf('-----------------------------------------------------------------------\n');
fprintf('LF detector, 1550nm, LG00\n');
print_parameters(1550e-9,ftest,Ltest);

end

function print_parameters(lambda,f,Lpr)
% cavity length 
L=10e3;
c=299792458.0;

if (lambda==1550e-9)
  % mirror diameter
  D=0.45;
  w=D/2/2.63;
  LGp=0;
  LGl=0;
else
  % conversion factor for LG33 : 2*4.31
  D=0.62;
  w=D/2/4.31;
  LGp=3;
  LGl=3;
end

loss=FT_LG_clipping_loss(LGp,LGl,w,D/2);

fprintf(' arm cavity -------------------\n');
fprintf('  cavity length: %.2f m\n',L);
fprintf('  mirror diameter: %.2f cm\n',D*100);
fprintf('  -------------------\n');
fprintf('  beam radius (for 1ppm loss): %.2f mm\n',w*1000);
fprintf('  (check clipping loss per mirror: %.2f ppm)\n',loss*1e6);
fprintf('  -------------------\n');

if (lambda==1550e-9)
  w=0.09;
else
  w=0.072;
end

loss=FT_LG_clipping_loss(LGp,LGl,w,D/2);

fprintf('  actual beam radius: %.2f mm\n',w*1000);
fprintf('  actual clipping loss per mirror: %.2f ppm\n',loss*1e6);


% round to a mm
w=round(w*1000)/1000;

% compute Rc for symmetric cavity
[Rct]=FT_cavity_w2Rc(w,L,lambda);

% round radius of curvature to rL meters
rL=1;
Rc=round(Rct(2)/rL)*rL;

fprintf('  -------------------\n');
fprintf('  mirror Rc: %.2f m\n',Rc);

% compute the rest of the beam parameters:
g=1-L/Rc;
[w0arm,zarm,zrarm,w1arm,w2arm,qarm]=FT_cavity_gg2w0z(g,g,L,lambda);

fprintf('  cavity eigenmode:\n');
fprintf('  waist size: %.2f mm\n',w0arm*1e3);
fprintf('  waist position: %.2f m\n',zarm);
fprintf('  spot size mirror 1: %.2f mm\n',w1arm*1e3);
fprintf('  spot size mirror 2: %.2f mm\n',w2arm*1e3);
fprintf('  Rayleigh range: %.2f m\n',zrarm);

fprintf(' PR cavity --------------------\n');
% Now focus the beam through a lens in the ITM
%f=310;
%Lpr=310;

fprintf('  focal length of ITM cavity: %.2f m\n',f);
fprintf('  Length of PR cavity: %.2f m\n',Lpr);

[Mlens] = FT_ABCD_lens(f);
q2=FT_q_ABCD_q(Mlens,qarm,1,1);
zr2=FT_q_to_zr(q2);
[w02,z02] = FT_q_to_w0z(q2,lambda,1);

q3=q2+Lpr-10;
w3=FT_q_to_w(q3,lambda,1);
fprintf('  spot size on BS: %.2f mm\n',w3*1e3);

q4=q3+10;
w4=FT_q_to_w(q4,lambda,1);
fprintf('  spot size on PR: %.2f mm\n',w4*1e3);

Rcpr=FT_q_to_Rc(q4);
fprintf('  Rc of PR: %.2f m\n',Rcpr);


Rcfake=-1.0*FT_q_to_Rc(q2);
% the pr cavity is equivalent to a cavity with
% R1=Rcpr and R2=Rcfake

FSRpr=c/2/Lpr;

gpr1=1-Lpr/Rcpr;
gpr2=1-Lpr/Rcfake;
gouy=1/pi*acos(sqrt(gpr1*gpr2));
gouydeg=gouy/pi*180;

mod_sep=gouy*FSRpr/pi;
fprintf('  FSR of PR: %.2f kHz\n',FSRpr/1e3);
fprintf('  mode sep of PR: %.2f kHz\n',mod_sep/1e3);
fprintf('  Gouy phase of PR: %.2f deg\n',gouydeg);
fprintf('  Rayleigh range of PR: %.2f m\n',zr2);

end