%
%--------------------------------------------------------------------------
% function [z_min,z_max]=FT_plot_virgo_map(map,r)
%
% A function for Matlab which plots a mirror surface map as read by 
% FT_read_virgo_map from the original map file in the standard
% virgo format. The function returns the maximum deviation from
% zero in nanometers. If a radius is given, this is computed only in
% the area of radius r around the center. The colorscale is limited to
% z_min, z_max.
%
% filename: name of virgo file
% map:      the surface map (variable converted to 'map' structure by
%           FT_read_virgo_map)
% r:        radius for which the deviation from zero should be computed,
%           use r=0 for the whole mirror
%
%
% z_min, z_max: maximum deviation from zero in the area defined
%               by radius r
%
% Charlotte Bond, andreas Freise 05.09.2008
%--------------------------------------------------------------------------
%

function [z_min,z_max] = FT_plot_virgo_map(map,r)

baseid='plot_virgo_map';

% Set the name of the map to the name of the data and remove underscores
% and file type.
name=FT_remove_file_extension(map.name);
name2=FT_replace_underscores(name,'\_');


[rows,cols]=size(map.data);

% Set the title for the top of the plot
win_title=sprintf('Virgo mirror map: %s', name );
fig_title=sprintf('Virgo mirror map: %s', name2 );

figure('name', win_title )

% Define the x and y scales in [cm]
x_index=linspace(1,cols,cols);
y_index=linspace(1,rows,rows);
x=(x_index-(map.x0))*map.xstep*100;
y=(y_index-(map.y0))*map.ystep*100;

% Rescaling map into nanomater
% (this should the default but its better to make sure
% so that the coloraxis label is correct)
data=map.data*map.scaling*1e9;;

% set outside values to 0 for plotting
%data(map.nan)=0;

% Plot a surface plot of the map
surf(x,y,data,'Edgecolor','none');

% Set the figure to give a 2D view of the surface plot
view(2);

% Set the y and x scales to linear and attach axis labels
set(gca, 'YScale', 'lin');
set(gca, 'XScale', 'lin');
xlabel('x [cm] ');
ylabel('y [cm] ');
zlabel('Height [nm]');
axis square

% Reduce the plotted area to size of mirror:
%  find all points that are not NaN
[ridx,cidx]=find(map.notnan);
%  and find the min/max x/y values for this data set
x_min=min(x(cidx));
x_max=max(x(cidx));
y_min=min(y(ridx));
y_max=max(y(ridx));
set(gca, 'XLim', [x_min x_max]);
set(gca, 'YLim', [y_min y_max]);
disp(sprintf('Zoom: plotting only (rows,cols): (%d, %d) from a (%d, %d) data set',max(ridx)-min(ridx),max(cidx)-min(cidx),rows,cols));

% Insert colorbar in figure
% the following two lines worked in previous Matlab versions, but not anymore with 2009
%hc=colorbar('Xaxislocation','top');
%set(get(hc,'xlabel'),'string','Height [nm]','fontsize',15,'Fontname','helvetica','fontweight','bold');
hc=colorbar;
ylabel(hc,'Height [nm]');


if (r==0)
  z_min=min(min(data));
  z_max=max(max(data));
  disp(sprintf('Maximum deviations from zero: %g  <--> %g nm',z_min,z_max));
else
  [idx, npoints]=FT_select_area_in_map(map,r);
  z_min=min(min(data(idx)));
  z_max=max(max(data(idx)));
  disp(sprintf('Maximum deviations from zero (within circle of r=%d): %g  <--> %g nm',r, z_min,z_max));
end


% Make a symmetric color mapping
c_max=max(abs([z_max z_min]));
set(gca, 'CLim', [-c_max c_max]);
set(gca, 'ZLim', [-c_max c_max]);

cmap=flip_colormap;
colormap(cmap);

% Set a tile and align it to the left in order to leave enough space
% for the colorbar label
ht=title( fig_title );
pos=get(ht,'Position');
pos(1)=x_min;
set(ht,'Position',pos);
set(ht,'HorizontalAlignment','left')

grid off;