%
%--------------------------------------------------------------------------
% function [map,x_tilt,y_tilt,Rc_out,offset]=FT_convert_virgo_map_for_finesse(filename,invert,Rc_in,w)
%
% A function for Matlab which reads a mirror surface map in the standard
% virgo format. It then inverts the map if specified and then fits and 
% removes any tilt of the surface. If the mirror is curved Matlab attempts 
% to fit and remove the curvature. 
% Matlab then writes the edited map for finesse, in the standard finesse
% structure. In addition, it produces an amplitude map that describes
% the aperture given by the mirror map; the amplitude map is saved as 'filename_map.txt'.
%
% filename: name of virgo file
% invert:   if mirror is inverted (with substrate marking at the top) for
%           use in finesse. bit coded integer:
%             0: do not invert
%             1: invert horizontally
%             2: invert vertically
%             3: invert in both directions
% Rc_in:    initial guess for radius of curvature [m].  If mirror is not 
%           curved enter Rc_in=0.
% w:        beam radius on mirror [m] for use in the weighting function used in 
%           the tilt and curvature removal functions.  Value can be set
%           manually or it can take the set values for a particular type of
%           mirror.
%           Type of mirror: End mirror    -> w='end'
%                           Input mirror  -> w='input'
%
% map:      map structure, the data hold sthe surface maps after removal of 
%           curvature, piston and offset
% x/y_tilt: returns tilts along x/y direction which have been removed from the
%           surface map [rad]
% Rc_out:   radius of curvature of fitted sphere which has been removed
%           from the map [m]
% offset:   the value set as the new zero point of the surface map (value
%           removed from surface map) [nm]
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond, Andreas Freise  05.09.2008
%--------------------------------------------------------------------------
%

% Description: converts a Virgo mirror map into Finess readable format
% Keywords: mirror, map, Virgo, Finesse

function [map3,x_tilt,y_tilt,Rc_out,offset] = FT_convert_virgo_map_for_finesse(filename,invert,Rc_in,w)
  
  baseid='convert_virgo_map_for_finesse';

  tic
  % read map
  map=FT_read_virgo_map(filename);

  % set display style for fitting routines to 'final'
  display_style=0;
  
  % prepare output filename
  basename=FT_remove_file_extension(filename);
  map_name=sprintf('%s', basename, '_for_finesse.txt');
  amp_map_name=sprintf('%s', basename, '_aperture.txt');
  result_name=sprintf('%s', basename, '_conversion_info.txt');

  FT_message(' --- recentering map ...');
  % recenter map
  map=FT_recenter_map(map);

  FT_message(' --- removing offset ...');
  % average 1mm region in the center and remove average as offset
  [map,offset]=FT_remove_offset_from_virgo_map(map,1e-2);
    
  % Select default value of w, in meters
  if (strcmp(w,'end'))
      w=55e-03;
  elseif (strcmp(w,'input'))
      w=20e-03;
  end
 
  FT_message(' --- removing curvature ...');
  % remove curvature
  if (Rc_in~=0)
    [map2,Rc_out, deltax, deltay]=FT_remove_curvature_from_virgo_map(map,Rc_in,w, display_style);
  else
    map2=map;
    Rc_out=0;
  end

  FT_message(' --- removing piston ...');
  % remove piston
  [map3,x_tilt,y_tilt,offset2]=FT_remove_piston_from_virgo_map(map2,w, display_style);        
    
  offset=offset+offset2;
  
  if (invert)
    map3=FT_invert_virgo_map(map3, invert);
  end
  
  % Write map to file in finesse format
  FT_message(sprintf('saving file "%s" ...\n',map_name));
  FT_write_surface_map(map_name,map3);
  
  %create aperture map
  map4=FT_virgo_aperture_map(map3);

  % Write aperture map to file in finesse format
  FT_message(sprintf('saving file "%s" ...\n',amp_map_name));
  FT_write_surface_map(amp_map_name,map4);

  % Write results to STDOUT
  FT_write_map_fit_results(1,map3,Rc_out,x_tilt,y_tilt,offset,w,invert);

  % ... and to file
  FT_message(sprintf('writing results to file "%s" ...\n',result_name));
  FT_make_backup(result_name);
  [FID, result]=fopen(result_name,'w');  
  if (FID==-1)
    msgid=[baseid,':fileopen'];
    error(msgid,result);
  end
  FT_write_map_fit_results(FID,map3,Rc_out,x_tilt,y_tilt,offset,w,invert);
  fclose(FID);
  
  % and maybe plot the result
  FT_plot_virgo_map(map3,0);

  toc
  
