%
%--------------------------------------------------------------------------
% function [map_out,x_tilt,y_tilt,Rc_out,offset]=FT_convert_ligo_map_for_finesse(filename,invert,Rc_in,w)
%
% A Matlab function to convert a raw ligo mirror map for use in finesse.
% The code removes tilts from the surface and curvature if required.  Based
% on FT_convert_virgo_map_for_finesse.  The code outputs to maps, the
% phase map and an amplitude map representing the aperture of the mirror,
% i.e. totally absorbing outside the optical surface.
%
% filename: name of ligo file
% invert:   if mirror is inverted (with substrate marking at the top) for
%           use in finesse. bit coded integer:
%             0: do not invert
%             1: invert horizontally
%             2: invert vertically
%             3: invert in both directions
% Rc_in:    initial guess for radius of curvature [m].  If mirror is not 
%           curved enter Rc_in=0.
%           Can enter strings to specify removal of astigmatism or residual
%           curvature.  These use a convolution with Zernike polynomials.
%           'all':  remove curvature and astigmatism
%           'astigmatism':  just remove astigmatism
%           'residual': just remove curvatue
%
% w:        beam radius on mirror [m] for use in the weighting function used in 
%           the tilt and curvature removal functions.  Value can be set
%           manually or it can take the set values for a particular type of
%           mirror.
%           Type of mirror: End mirror    -> w='end' = 6.2 cm (AdLIGO)
%                           Input mirror  -> w='input' = 5.3 cm (AdLIGO)
%
% map:      map structure, the data holding the surface maps after removal of 
%           curvature, piston and offset
% x/y_tilt: returns tilts along x/y direction which have been removed from the
%           surface map [rad]
% Rc_out:   radius of curvature of fitted sphere which has been removed
%           from the map [m]
% offset:   the value set as the new zero point of the surface map (value
%           removed from surface map) [nm]
%
% Charlotte Bond Andreas Freise  17.12.2010
%--------------------------------------------------------------------------


function [map_out,x_tilt,y_tilt,Rc_out,offset] = FT_convert_ligo_map_for_finesse(filename,invert,Rc_in,w)


  baseid='convert_ligo_map_for_finesse';

  % read map
  map=FT_read_ligo_map(filename);

  % set display style for fitting routines to 'final'
  display_style=0;
  
  % prepare output filename
  basename=FT_remove_file_extension(filename);
  basename=FT_remove_phrase_from_string(basename,'_asc');
  map_name=sprintf('%s', basename, '_for_finesse.txt');
  amp_map_name=sprintf('%s', basename, '_aperture.txt');
  result_name=sprintf('%s', basename, '_conversion_info.txt');

  disp(' --- recentering map ...');
  % recenter map
  map=FT_recenter_mirror_map(map);

  % Select default value of w, in meters
  % Need to reset these for the ligo set-up
  if (strcmp(w,'end'))
      w=62e-03;
  elseif (strcmp(w,'input'))
      w=53e-03;
  end
 
  disp(' --- removing curvature ...');
  % remove curvature
  if (Rc_in~=0)
      if (strcmp(Rc_in,'all') || strcmp(Rc_in,'residual') || strcmp(Rc_in,'astigmatism'))
        [map2,Rc_out]=FT_remove_zernike_curvatures_from_map(map,Rc_in); 
      else
        [map2,Rc_out, deltax, deltay]=FT_remove_curvature_from_mirror_map(map,Rc_in,w, display_style);
      end
  else
    map2=map;
    Rc_out=0;
  end
  
  disp(' --- removing offset ...');
  % average 1mm region in the center and remove average as offset
  [map2,offset]=FT_remove_offset_from_mirror_map(map2,1e-2);
  
  disp(' --- removing piston ...');
  % remove piston
  [map3,x_tilt,y_tilt,offset2]=FT_remove_piston_from_mirror_map(map2,w, display_style);        
    
  offset=offset+offset2;
  
  if (invert)
    map3=FT_invert_mirror_map(map3, invert);
  end
  
  map_out=map3;
  
  % Write map to file in finesse format
  disp(sprintf(' --- saving file "%s" ...\n',map_name));
  FT_write_surface_map(map_name,map_out);
  
  %create aperture map
  amp_map=FT_create_aperture_map(map_out);

  % Write aperture map to file in finesse format
  disp(sprintf(' --- saving file "%s" ...\n',amp_map_name));
  FT_write_surface_map(amp_map_name,amp_map);
  
  % Write results to STDOUT
  write_results(1,map_out,Rc_out,x_tilt,y_tilt,offset,w,invert);

  % ... and to file
  disp(sprintf(' --- writing results to file "%s" ...\n',result_name));
  FT_make_backup(result_name);
  [FID, result]=fopen(result_name,'w');  
  if (FID==-1)
    msgid=[baseid,':fileopen'];
    error(msgid,result);
  end
  write_results(FID,map_out,Rc_out,x_tilt,y_tilt,offset,w,invert);
  fclose(FID);
  
  % and maybe plot the result
  FT_plot_mirror_map(map_out,0);

end
  
function []=write_results(FID,map,Rc,x_tilt,y_tilt,offset,w,invert);
  
  % use FID=1 for stdout
  fprintf(FID,'\n---------------------------------------------------------\n');
  fprintf(FID,'Map: %s\n',map.name);
  fprintf(FID,'Date: %s\n', datestr(now,'dd. mmmm yyyy (HH:MM:SS)'));
  fprintf(FID,'---------------------------------------------------------\n');
  fprintf(FID,'Radius of curvature Rc=%4.3f m\n',Rc);
  fprintf(FID,'Angles: x_tilt=%1.4g, y_tilt=%1.4g rad\n',x_tilt,y_tilt);
  fprintf(FID,'Offset: %3.3f nm\n',offset);
  fprintf(FID,'Weighting function used with w=%3.3f mm\n',w*1e3);
  if (invert)
    fprintf(FID,'Map flipped ');
    if (bitand(invert,1))
      fprintf(FID,'horizontally');
    end
    if (bitand(invert,1) && bitand(invert,2))
      fprintf(FID,' and vertically\n');
    elseif (bitand(invert,2))
      fprintf(FID,'vertically\n');
    end
  end
  fprintf(FID,'---------------------------------------------------------\n\n');

end

