%
%--------------------------------------------------------------------------
% function [z_min,z_max]=FT_plot_mirror_map(map,r)
%
% A function for Matlab which plots a mirror surface map in the standard
% map structure.  This function is based on the original function, 
% FT_plot_mirror_map.  The function returns the maximum deviation from
% zero in nanometers. If a radius is given, this is computed only in
% the area of radius r around the center. The colorscale is limited to
% z_min, z_max.
%
% map:      the surface map 
% r:        radius for which the deviation from zero should be computed,
%           use r=0 for the whole mirror
%
% z_min, z_max: maximum deviation from zero in the area defined
%               by radius r
%
% Part of the Simtools package, http://www.gwoptics.org/simtools
% Charlotte Bond, Andreas Freise 05.09.2008
%--------------------------------------------------------------------------
%

% Description: plots a mirror surface map
% Keywords: plot, mirror, map

function [z_min,z_max] = FT_plot_mirror_map(map,r)

baseid='plot_mirror_map';

% set default for r if not given as argument
if nargin == 1
    r = 0;
end

% Set the name of the map to the name of the data and remove underscores
% and file type.
name=FT_remove_file_extension(map.name);
name2=FT_replace_underscores(name,'\_');

[rows,cols]=size(map.data);

% Set the title for the top of the plot
win_title=sprintf('Mirror map: %s', name );
fig_title=sprintf('Mirror map: %s', name2 );

figure('name', win_title )

% Define the x and y scales in [cm]
x_index=linspace(1,cols,cols);
y_index=linspace(1,rows,rows);
x=(x_index-(map.x0))*map.xstep*100;
y=(y_index-(map.y0))*map.ystep*100;

data=map.data;

if (map.type==0) % phase map
% Rescaling map into nanomater
  data=data*map.scaling*1e9;
  map_zlabel='Height [nm]';
elseif (map.type==1) % absorption
  map_zlabel='absorption';
elseif (map.type==2) % reflection
  map_zlabel='reflectivity';
else
  msgid=[baseid,':maptype'];
  result='map has an unknown type';
  error(msgid,result);
end
  
% set outside values to 0 for plotting
%data(map.nan)=0;

% Plot a surface plot of the map
%surf(x,y,data,'Edgecolor','none');

% Set the figure to give a 2D view of the surface plot
%view(2);

% I can't see why the 3D renderer surf is needed here
% changed to imagesc 2d plot which seems a better option
imagesc(x,y,data);

% Set the y and x scales to linear and attach axis labels
set(gca, 'YScale', 'lin');
set(gca, 'XScale', 'lin');
xlabel('x [cm] ');
ylabel('y [cm] ');
zlabel(map_zlabel);
axis square
% flip y axis for imagesc plot
set(gca,'YDir','normal')

% Reduce the plotted area to size of mirror:
%  find all points that are not NaN
[ridx,cidx]=find(map.notnan);
%  and find the min/max x/y values for this data set
x_min=min(x(cidx));
x_max=max(x(cidx));
y_min=min(y(ridx));
y_max=max(y(ridx));
set(gca, 'XLim', [x_min x_max]);
set(gca, 'YLim', [y_min y_max]);
disp(sprintf('Zoom: plotting only (rows,cols): (%d, %d) from a (%d, %d) data set',max(ridx)-min(ridx),max(cidx)-min(cidx),rows,cols));

% Insert colorbar in figure
% the following two lines worked in previous Matlab versions, but not anymore with 2009
%hc=colorbar('Xaxislocation','top');
%set(get(hc,'xlabel'),'string','Height [nm]','fontsize',15,'Fontname','helvetica','fontweight','bold');
hc=colorbar;
ylabel(hc,map_zlabel);

if (r==0)
  z_min=min(min(data));
  z_max=max(max(data));
  disp(sprintf('Maximum deviations from zero: %g  <--> %g nm',z_min,z_max));
else
  [idx, npoints]=FT_select_area_in_map(map,r);
  z_min=min(min(data(idx)));
  z_max=max(max(data(idx)));
  disp(sprintf('Maximum deviations from zero (within circle of r=%d): %g  <--> %g nm',r, z_min,z_max));
end

c_max=max(abs([z_max z_min]));
  
if (map.type==0) % phasemap
  % Make a symmetric color mapping
  cmap=FT_flip_colormap;
  set(gca, 'CLim', [-c_max c_max]);
  set(gca, 'ZLim', [-c_max c_max]);
else
  %  cmap=FT_neg_colormap;
  cmap=1-gray;
  set(gca, 'CLim', [z_min z_max]);
  set(gca, 'ZLim', [z_min z_max]);
end

%# size of colormap
n = size(cmap,1);
%# color step
dmap=(z_max-z_min)/n;

%# add nan color to colormap
%colormap(cmap);
colormap([[1 1 1]; cmap]);

%# changing color limits
caxis([-c_max-dmap c_max]);

%# change Y limit for colorbar to avoid showing NaN color
ylim(hc,[z_min z_max])

% Set a tile and align it to the left in order to leave enough space
% for the colorbar label
ht=title( fig_title );
pos=get(ht,'Position');
pos(1)=x_min;
set(ht,'Position',pos);
set(ht,'HorizontalAlignment','left')

grid off;

end