%
%--------------------------------------------------------------------------
% function [R,Z]=FT_create_sphere_for_map(map,Rc,x0,y0,offset,xbeta, ybeta)
%
% A function for Matlab which takes the x,y vectors from a Virgo mirror map 
% and computes a spherical surface in nanomaters, for a given radius of
% curvature Rc, an offset on the z-axis 'offset' and tilting angles
% 'xbeta', 'ybeta'.
%
% map   :  the surface map (variable converted to 'map' structure by
%          FT_read_virgo_map)
% Rc     : radius of curvature [m]
% x0,y0  : center of the map data array [data points] (float, not integer!)
% offset : offset along the z-axis [nm]
% x_tilt : tilt along x (i.e. around the y-axis) [rad]
% y_tilt : tilt along y (i.e. around the x-axis) [rad]
%
% Rsq    : array of the dimension of map, with the radial distance squared 
%          from the center [m^2]
% Z      : array of the dimension of map, with the z-distance from the 
%          center [nm]
%
% Andreas Freise 05.09.2008
%--------------------------------------------------------------------------
%

function [Rsq,Z]=FT_create_sphere_for_map(map,Rc,x0,y0,offset,x_tilt, y_tilt)
  
  [rows,cols]=size(map.data);
  newx=(linspace(1,cols,cols)-x0)*map.xstep;
  newy=(linspace(1,rows,rows)-y0)*map.ystep;
  [X,Y] = meshgrid(newx,newy);
  Rsq = X.^2+ Y.^2;

  Z=offset+(X*tan(x_tilt)+Y*tan(y_tilt))/map.scaling;

  if (Rc>0)
    Z = Z+(Rc-sqrt(Rc^2 - Rsq))/map.scaling;
  end
  
  if (Rc<0)
    Z = Z-(abs(Rc)-sqrt(Rc^2 - Rsq))/map.scaling;
  end