%--------------------------------------------------------------------------
% function [map] = FT_create_elliptical_aperture_map(D,alphax,alphay,realsize,gridsize)
%
% A function for Matlab which creates an elliptical aperture map in the
% standard map format.  The function works to create the elliptical
% aperture seen by a beam when a mirror is under a certian angle. (for a
% circle use alphax=alphay=0).
% The function saves the aperture map in a file in the standard Finesse map
% format.
%
% D:        Mirror diameter [m]
% alphax:   Angle of incidence (x direction) [degrees]
% alphay:   Angle of incidence (y direction) [degrees]
% realsize: Overall square size of map [m]
% gridsize: Number of x and y points.
%
% Charlotte Bond    08.08.2011
%--------------------------------------------------------------------------
%

function [map] = FT_create_elliptical_aperture_map(D,alphax,alphay,realsize,gridsize)

    % Find ellipse dimensions
    a=D*cos(alphax*pi/180)/2;
    b=D*cos(alphay*pi/180)/2;
    
    map_name=sprintf('aperture_map_%gmm_%gmm.txt',round(a*1000),round(b*1000));
    
    % Size of map
    cols=gridsize;
    rows=gridsize;

    % Optical centre
    x0=(cols+1)/2;
    y0=(rows+1)/2;
    
    % Type 1: absorption map
    % Field 0: effects both transmitted and reflected light
    type=1;
    field=0;
    
    scaling=1;
    
    % Set step size
    xstep=realsize/(cols-1);
    ystep=realsize/(rows-1);
    
    % Define x and y coordinates from the map centre
    x=(1-x0:cols-x0)*xstep;
    y=(1-y0:rows-y0)*ystep;
    
    % Fill in aperture map for ellipse, i.e. 1 (total absoprtion) outside
    % ellipse and 0 (no absortion) inside it.
    for i=1:length(x)
        for j=1:length(y)
            
            if (x(i)^2/a^2)+(y(j)^2/b^2)<=1
                data(j,i)=0;
                
            else
                data(j,i)=1;
            end
        end
    end
    
    % Create new map
    map=FT_new_surface_map(map_name,data,type,field,x0,y0,xstep,ystep,scaling);
    
    % Write map in Finesse format
    FT_write_surface_map(map_name,map);
    
    % Plot map
    FT_plot_mirror_map(map,0)


end

