%
%--------------------------------------------------------------------------
% FTE_finesse_ad_nm.m
%
% Test example that uses FT_mode_content and related functions with a
% Finesse output file in which the beam amplitude has been generated
% with the `ad' detectors.
%
% This file requires file `test_ad_nm.kat'.
%
% Part of the SimTools package
% Andreas Freise  12.02.2011
%--------------------------------------------------------------------------
%

% Description: Example for using SimTools with Finesse: get mode coefficients of output beam
% Keywords: finesse, example, simulation, kat, amplitude, mode, coefficient, Gauss


clear all;

% define list of frequencies for which the mode decomposition should be done
frequency_vector=[0];

% max mode order
max_mode_order=4;

% name of kat file which contains 'blocks'
inname='test_ad_nm.kat';

% name of file that we will create now from this script
katname='mytest.kat';

% read in block from testblock.kat
block=FT_read_blocks_from_file(inname);

% now selectively copy some blocks from the list 'block' over
% into the list 'outbl'.
% we can access the block usually by name or by index:
outbl(1)=FT_copy_block(block,'laser'); 
outbl(2)=FT_copy_block(block,'cavity');
outbl(3)=FT_copy_block(block,'maxtem');
outbl(4)=FT_copy_block(block,'attributes');
outbl(5)=FT_copy_block(block,'plotcommands');

% create new block containing all amplitude detectors
tmp_bl=FT_create_new_block('det');
for l=1:length(frequency_vector)
  f=frequency_vector(l);
  for n=0:max_mode_order
    for m=0:4-n
      tmp_str=sprintf('ad ad_%d_%d%d %d %d %g n1',l-1,n,m,n,m,f);
      tmp_bl=FT_add_line_to_block(tmp_bl,tmp_str);
    end
  end
end
tmp_bl=FT_add_line_to_block(tmp_bl,'yaxis re:im');
tmp_bl=FT_add_line_to_block(tmp_bl,'noxaxis');
outbl(6)=tmp_bl;

% create FT variable, which contains the path to 'kat'
FT=FT_init_Finesse('name','kat','path','~/work/bin/');
FT=FT_check_for_kat_binary(FT);

% write all blocks from list 'outbl' to file 'mytest.kat'
FT_write_blocks_into_file(outbl,katname);

% run Finese and read data
Run=FT_create_new_kat_run();
Run.filename=katname;
Run=FT_run_kat_simulation(FT,Run);

% get coefficients as complex numbers
for l=1:length(Run.y)/2;
  coeff(l)=Run.y(2*l-1)+i*Run.y(2*l);
end

% now sort coefficients into `gc' structures
cindex=1;
for l=1:length(frequency_vector)
  [gc(l),n_coeff]=FT_init_gauss_coefficients('HG', max_mode_order);

  % normalise coefficients by total power
  cindex1=1+n_coeff*(l-1);
  cindex2=n_coeff*l;
  power=sum(coeff(cindex1:cindex2).*conj(coeff(cindex1:cindex2)));
  coeff(cindex1:cindex2)=coeff(cindex1:cindex2)/sqrt(power);

  for n=0:max_mode_order
    for m=0:4-n
      [gc(l)]=FT_set_mode_coefficient(gc(l), coeff(cindex), n, m);
      cindex=cindex+1;
    end
  end
end
  
% print result
for l=1:length(frequency_vector)
  fprintf('---------------------------------------------\n');
  fprintf('Mode distribution for frequency %g:\n',frequency_vector(l));
  FT_print_gauss_coefficients(gc(l),1e-3,0);
end