%---------------------------------------------------------------------  
% function [FT] = FT_init_Finesse(varargin)
% 
% Creates a new 'FT' struture
% You can set the name+path for the Finesse binary with this function.
% Please run FT_check_for_kat_binary(FT) after calling this function.
%
% Running with no arguments defaults to using the environment variable
% FINESSE_DIR for kat.exe location.
%
% Usage for example:
% FT=FT_init_Finesse('name','kat','path','~/work/bin/');
% or
% FT=FT_init_Finesse('name','kat.exe')
% or
% FT=FT_init_Finesse()
%
% FT: structure that contains path and name of Finesse executable 
%
% Part of the SimTools package
% Andreas Freise 02.02.2010 afreise@googlemail.com
%---------------------------------------------------------------------  

% Description: Creates a new "FT" structure
% Keywords: create, new, FT, structure

function [FT] = FT_init_Finesse(varargin)

  baseid='init_Finesse';
  
  % set defaults:
  tpath='';
  tname='';
  
  % ---- parse input arguments
  if nargin > 1
    % check which calling syntax is used
    actual_n=1;
    while actual_n<=nargin
      switch lower(varargin{actual_n})
       case 'path',
        tpath=varargin{actual_n+1};
       case 'name',
        tname=varargin{actual_n+1};
       otherwise ,
        msgid=[baseid,':checkinarg'];
        result=sprintf(['Unknown option: ', varargin{actual_n}, '.\n']);
        error(msgid,result);
        return
      end;   
      actual_n=actual_n+2;
    end; 
  elseif nargin ==0
    if isempty(getenv('FINESSE_DIR'))
        error('The FINESSE_DIR environment variable was not set. Please run the command "setenv(''FINESSE_DIR'',''Finesse directory'')" or add it to your startup.m file.');
    else
        tpath = getenv('FINESSE_DIR');
    end
  end
  
  % ----- checking whether path is consistent -----
  if (length(tpath)>0) 
    if (((~isempty(strfind(tpath,'\'))) && (length(tpath)~=max(strfind(tpath,'\')))) || ((~isempty(strfind(tpath,'/'))) && (length(tpath)~=max(strfind(tpath,'/')))))
    msgid=[baseid,':checkinarg'];
    result=sprintf('Path "%s" must end with "\\\\" or "/".\n', tpath);
    error(msgid,result);
    end
  end

  % full path to Finesse executable
  FT.path=tpath; 

  
  % ----- check if Finesse binary exists ----
  found=0;
  if (isempty(tname))
    for tmpname = [{'kat'},{'kat.exe'}] 
      tname=cell2mat(tmpname);
      FT.katbinary=[tpath,tname];
      FT.name=tname;
      found=FT_check_for_kat_binary(FT);
      if (found==1)
        break;
      end;
    end
  else  
    FT.katbinary=[tpath,tname];
    FT.name=tname;
    found=FT_check_for_kat_binary(FT)
  end
  
  if (found==0)
      msgid=[baseid,':init'];
      result=['cannot run "kat" or "kat.exe" at path ',tpath];
      error(msgid,result);    
  end
      
  % don't change anything below this line
  FT.katbinary=[tpath,tname];

  % ---- now for kat.ini ------
  % check if local kat.ini exists
  local=exist('./kat.ini');
  
  if (local==2)
    FT.katini='./kat.ini';
    disp(['Using local kat.ini: ',FT.katini]);
  else
    % check if global or local kat.ini exists
    FT.katini=getenv('KATINI');
    if (FT.katini)
      disp(['Found global kat.ini: ',FT.katini]);
    else
      if exist([FT.path,'kat.ini']);
        FT.katini=[FT.path,'kat.ini'];
        disp(['Using kat.ini in same path as executable: ',FT.katini]);        
      else
        disp('No kat.ini file found.');
      end
    end
  end
  
  