% ------------------------------------------------------
% function [coefficients, ns, ms] = FT_lg2hg (p,l)
% 
% Matlab/Octave function to compute the amplitude coefficients
% of Hermite-Gauss modes whose sum yields a Laguerre Gauss mode
% of order n,m. The formula is adpated from
% M.W. Beijersbergen et al 'Astigmatic laser mode converters and
% transfer of orbital angular momentum', Opt. Comm. 96 123-132 (1993)
%
% p:    Radial LG index
% l:    Azimuthal LG index
%
% The LG mode is written as u_pl with 0<=|l|<=p.
%
% The output is a series of coefficients for u_nm modes,
% given as complex numbers and respective n,m indices
% coefficients (complex vector): field amplitude for mode u_nm
% ns (int vector): n-index of u_nm
% ms (int vector): m-index of u_nm
%
% 20/11/2012:   Adapted coefficients to be compatible with Finesse Tools
%               definition of an LG mode (FT_LG_field.m), differs from
%               Beijersbergen by (-1)^p factor and has exp(il\phi) rather
%               than exp(-il\phi).  Also adapted for -l.
% 
% Andreas Freise, Charlotte Bond    25.03.2007
%----------------------------------------------------------------------------------

function [coefficients, ns, ms] = FT_lg2hg (p,l)

    % Mode order
    N=2*p+abs(l);

    % Indices for coefficients
    n = abs(l)+p;
    m = p;

    % create empty vectors
    coefficients=linspace(0,0,N);
    ns=coefficients;
    ms=coefficients;

    % Beijersbergen coefficients
    for j=0:N
        ns(j+1)=N-j;
        ms(j+1)=j;

        c=i^j * sqrt(factorial(N-j)*factorial(j)/(2^N*factorial(n)*factorial(m)));
        coefficients(j+1)=c*(-2)^j*FT_JacobiPol(n-j,m-j,j,0);
    end

    % If l is negative
    if l<0
        coefficients = conj(coefficients);
    end

    % Convert from LG definition in Beijersbergen93 to our definition
    % (FT_LG_field.m)
    coefficients = (-1)^p * conj(coefficients);


end