%---------------------------------------------------------------------  
% function [gp] = FT_update_gauss_param(gp,q,nr)
% 
% Matlab function that computes the parameters of 
% a Gaussian beam such as beam size, waist position etc
% from the Gaussian beam parameter `q'. 
%
% This function should be used if a structure `gp' has been created 
% before with FT_init_gauss_param.m. This function
% (re-computed) and stores the derived parameters from the
% q parameter in the structur `gp'.
% 
% This function can also be used to update the structure gp; the most 
% obvious use is for propagating the beam
% through free space, i.e. a propagation over 12 meters would be
%  gp=FT_update_gauss_param(gp,gp.q+12,gp.nr)
%
% gp (structure):  structure to store beam parameters, 
%                  see FT_init_gauss_param
% q (complex):     vector of two complex numbers with q(1)=q_x and 
%                  q(2)=q_y and q_x,q_y the Gaussian beam parameter 
%                  in the x-z plane and the y-z plane respectively
% nr (real):       index of refraction
% 
% Part of the SimTools package
% Andreas Freise, 19.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  

% Description: Computes the parameters of a Gaussian beam from the beam parameter
% Keywords: Gaussian, parameter, q, update

function [gp] = FT_update_gauss_param(gp,q,nr)
  
  baseid='update_gauss_param';

  if (isreal(q))
    msgid=[baseid,':checkinarg'];
    result='"q" must be complex';
    error(msgid,result);
  end
  
  % store new parameters in gp
  if length(q)==1
    gp.q(:)=q;
    gp.is_astigmatic=0;
  elseif (length(q)==2)
    gp.q=q;
    if (q(1)==q(2))
      gp.is_astigmatic=0;
    else
      gp.is_astigmatic=1;
    end
  else
    msgid=[baseid,':checkinarg'];
    result='"q" must be a complex number or a vector of two cmplex numbers';
    error(msgid,result);
  end
  gp.nr=nr;
  
  % recompute derived parameters
  gp.q0=1i*imag(gp.q);
  [gp.w0,gp.z] = FT_q_to_w0z(gp.q,gp.lambda,gp.nr);
  gp.zr=FT_q_to_zr(gp.q);
  gp.w=FT_q_to_w(gp.q,gp.lambda,gp.nr);
  gp.Rc=FT_q_to_Rc(gp.q);
  gp.Psi=FT_q_to_Psi(gp.q);