%---------------------------------------------------------------------  
% function [] = FT_help(help_name)
%
% Prints list of functions & descriptions matching input
% by name, keyword or category
%
% Part of the SimTools package, http://www.gwoptics.org/simtools
% Robert Unwin, Andreas Freise 11.02.11 
%---------------------------------------------------------------------  

function [] = FT_help(help_name, show_path)

  fid=1; % print to STDOUT by default
  
  %---------------------------------------------
  % for debugging
  %fid=fopen('help_out.txt','w');
  %print_usage(fid);
  %---------------------------------------------
  
  
  fprintf(fid,'\n');
  
  % print usage when no input argument is given
  if (nargin==0)
    print_usage(1);
    return;
  end

  if (nargin<2)
    show_path=false;
  end
      
  % open data files
  help_info_out;
  help_info_key;
  help_info_cat;
  
  if (strcmp(help_name,'all'))
    fprintf(fid,'Displaying all entries:\n\n');
    nfiles = length(index);
    for i=1:nfiles
      fprintf(fid,' - %s',index(i).name);
      fprintf(fid,': ');
      ndesc = length(index(i).description);
      for j=1:ndesc
        if (j~=1)
          fprintf(fid,'   ');
        end
        fprintf(fid,cell2mat(index(i).description(j)));
        fprintf(fid,'\n');
      end
      print_path(index(i).path,show_path);
    end
    fprintf(fid,'\n');
    return
  end
    
  if (strcmp(help_name,'key'))
    fprintf(fid,'Possible key words:\n\n');
    nkeywords = length(key);
    txt=' ';
    slength=0;
    first=true;
    for i=1:nkeywords
      if (~strcmp(key(i).word,'(no marker)') && ~strcmp(key(i).word,'(no keyword)')) 
        if (~first)
          txt=[txt,', '];
        end
        first=false;
        txt=[txt,lower(key(i).word)];
        slength=slength+length(key(i).word)+2;
        if (slength>60)
          fprintf(fid,'%s\n',txt);
          txt=' ';
          first=true;
          slength=0;
        end
      end
    end
      fprintf(fid,'%s\n',txt);
    return
  end
    
  Q = true; % define Boolean to flag whether match has been found

  % print function help if input argument is a full function name
  nfiles = length(index);
  for i=1:nfiles
    if strcmp(index(i).name,help_name);
      Q = false;
      fprintf(fid,'Function "%s":\n',help_name);
      ndesc = length(index(i).description);
      for j=1:ndesc
        fprintf(fid,cell2mat(index(i).description(j)));
        fprintf(fid,'\n');
      end
      fprintf(fid,'\n');
      help(help_name);
      print_path(index(i).path,show_path);
      fprintf(fid,'\n');
    end
  end

 shown=[];

  % check if help name is a keyword
  nkeywords = length(key);
  files = [];
  for i=1:nkeywords
    if strfind(lower(key(i).word),lower(help_name));
      Q = false;
      files = key(i).files;
      shown=[shown, files];
      break;
    end
  end

  if ~isempty(files)
    nfiles = length(files);
    fprintf(fid,'----------------------------------------------------------\n');
    fprintf(fid,'Displaying files containing keyword, "%s":\n\n',help_name);
    for i=1:nfiles
      fprintf(fid,' - %s',index(files(i)).name);
      fprintf(fid,': ');
      ndesc = length(index(files(i)).description);
      for j=1:ndesc
        if (j~=1)
          fprintf(fid,'   ');
        end
        fprintf(fid,cell2mat(index(files(i)).description(j)));
        fprintf(fid,'\n');
      end
       print_path(index(i).path,show_path);
    end
    fprintf(fid,'\n');
  end


  % check if help name is a category
  ncatwords = length(category);
  files = [];
  for i=1:ncatwords
    if strfind(lower(category(i).word),lower(help_name));
      Q = false;
      files = category(i).files;
      break;
    end
  end

  if ~isempty(files)
    nfiles = length(files);
    fprintf(fid,'----------------------------------------------------------\n');
    fprintf(fid,'Displaying files in category, "%s":\n\n',help_name);
    for i=1:nfiles
      if (isempty(find(files(i)==shown)))
        ndesc = length(index(files(i)).description);
        fprintf(fid,' - %s',index(files(i)).name);
        fprintf(fid,': ');
        print_path(index(i).path,show_path);
        for j=1:ndesc
          if (j~=1)
            fprintf(fid,'   ');
          end
          fprintf(fid,cell2mat(index(files(i)).description(j)));
          fprintf(fid,'\n');
        end
      end
    end
    fprintf(fid,'\n');
  end
  
  if Q
    fprintf(fid,'*** No matches found\n\n');
    print_usage;
  end
  
  if (fid>1)
    fclose(fid)
  end
end

function print_path(pathstring,show_path)
  if (show_path)
    fprintf(fid,'%s\n',pathstring);
  end
end
  
function []=print_usage(fid)
  fprintf(fid,'-------------------------------------------------------------------------\n');
  fprintf(fid,'SimTools 0.7, A. Freise, C. Bond 25.01.2013      Matlab utility functions\n');
  fprintf(fid,'\n');
  fprintf(fid,'Usage: FT_help(help_topic, [true])\n\n');
  fprintf(fid,'use the string "help_topic" to select a topic from the following list:\n\n');
  fprintf(fid,' - [keyword]: lists all functions with matching keyword\n');
  fprintf(fid,' - [category]: lists all functions in matching category\n');
  fprintf(fid,' - [filename]: prints a description of the function matching the filename\n');
  fprintf(fid,' - ''key'': lists all available keywords\n');
  fprintf(fid,' - ''all'': lists all available function from the SimTools package\n');
  fprintf(fid,'\n');
  fprintf(fid,'The second optional argument "true" causes the display of path information.\n');
  fprintf(fid,'\n');
end
