%---------------------------------------------------------------------  
% function [mygrid] = FT_init_grid(xpoints, ypoints, xsize, ysize, xoffset, yoffset)
% 
% Matlab function that creates a structure grid storing useful informations
% when dealing with data in x-y plane
% 
% xpoints (integer): number of data points in x direction
% ypoints (integer): number of data points in y direction
% xsize (real):      physical size of grid in x direction[m]
% ysize (real):      physical size of grid in y direction[m]
% xoffset (real):    offset if center of grid is not at x=0 [m]
% yoffset (real):    offset if center of grid is not at y=0 [m]
%
% mygrid (structure) : structure storing:
%                      xpoints, ypoints, xsize, ysize, xoffset, yoffset - as given by the user
%                      xstep, ystep - distance between two grid points [m]
%                      xaxis, yaxis - x and y vectors with phyiscal locationas of grid points [m]
%                      X,Y - output of  meshgrid(xaxis, yaxis)
%                      r_squared - output of X.^2+y.^2
%                      r   - output of  sqrt(r_squared)
%                      angle - output of  atan2(Y,X)
%                      xaxis_fft, yaxis_fft -  x and y vectors corresponding to a grid of fft2(mygrid)
%                      fft_X, fft_Y - output of meshgrid(xaxis_fft, yaxis_fft);
%                      fft_ir_squared - output of fftshift(fft_X.^2 + fft_Y.^2)
%
% Part of the SimTools package
% Andreas Freise, 19.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  

% Description: Creates a structure grid for storing useful information
% Keywords: create, grid, init

 function [mygrid] = FT_init_grid(xpoints, ypoints, xsize, ysize, xoffset, yoffset)

 mygrid.xpoints=xpoints;
 mygrid.ypoints=ypoints;
 mygrid.xsize=xsize;
 mygrid.ysize=ysize;
 mygrid.xoffset=xoffset;
 mygrid.yoffset=yoffset;
  
 mygrid.xstep=mygrid.xsize/mygrid.xpoints;
 mygrid.ystep=mygrid.ysize/mygrid.ypoints;
 
 xvector=(1:mygrid.xpoints)-1;
 yvector=(1:mygrid.ypoints)-1;
 
 mygrid.xaxis=-mygrid.xsize/2 + mygrid.xstep/2 + xvector*mygrid.xstep + mygrid.xoffset;
 mygrid.yaxis=-mygrid.ysize/2 + mygrid.ystep/2 + yvector*mygrid.ystep + mygrid.yoffset;
 
 [mygrid.X,mygrid.Y] = meshgrid(mygrid.xaxis,mygrid.yaxis);

 mygrid.r_squared = mygrid.X.^2 + mygrid.Y.^2;
 mygrid.r = sqrt(mygrid.r_squared);
 mygrid.angle = atan2(mygrid.Y,mygrid.X);

 % some extra params which are useful for FFT transformations
 mygrid.xaxis_fft = -1/(2*mygrid.xstep) + xvector/(mygrid.xpoints*mygrid.xstep);
 mygrid.yaxis_fft = -1/(2*mygrid.ystep) + yvector/(mygrid.ypoints*mygrid.ystep);

 % See function FT_FFT_propagate for where this is used
 [mygrid.fft_X,mygrid.fft_Y] = meshgrid(mygrid.xaxis_fft, mygrid.yaxis_fft);
 mygrid.fft_ir_squared=fftshift(mygrid.fft_X.^2+mygrid.fft_Y.^2);

 return
