%
%---------------------------------------------------------------------  
% function [result] = FT_FFT_results_cavity_fields(result, cavity, mygrid, laser, transmit)
% 
% Matlab function computing the reflected and transmitted field of a cavity
% when the input and circulating fields are known.
%
% cavity (structure):  cavity parameters and components 
%                      (see FT_init_cavity.m)
% mygrid (structure):  stores parameters of a 2D grid for FFT computations,
%                      see FT_init_grid.m
% laser (structure):   stores parameters of the laser such as wavelength, see
%                      FT_init_gauss_param.m
% transmit (integer):  number of the mirror (in cavity structure) that transmits 
%                      the beam (call this function several times if several transmitted
%                      fields are to be computed in multi-mirror cavities)
%
% result (structure):  storing results for a cavity scan and the power buildup
%                      this functions adds:
%                      - f_trans : field transmitted by the cavity (mirror no. "transmit")
%                      - f_leak  : field "leaking out' of the input mirror
%                      - f_refl  : field reflected by the input mirror
%
% Part of the SimTools package
% Andreas Freise, 26.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  
%

% Description: Computes the reflected and transmitted fields of a cavity
% Keywords: reflected, transmitted, field

function [result] = FT_FFT_results_cavity_fields(result, cavity, mygrid, laser, transmit)

  baseid='FFT_result_cavity_fields';
  
  clast=cavity.ncomponents;
  
  if (transmit<1 || transmit>clast)
    msgid=[baseid,':checkinarg'];
    result=sprintf('Number of transmitting mirror is %d but must be >0 and <%d.\n',transmit,clast);
    error(msgid,result);
  end

  % computing transmitted field
  result.f_trans = FT_FFT_propagate(result.f_circ, mygrid, laser.lambda, cavity.space(1).L, cavity.space(1).n);
  for p=2:transmit;
    result.f_trans = FT_FFT_mirror(result.f_trans,cavity.mirror(p-1),'front');
    result.f_trans = FT_FFT_propagate(result.f_circ, myrid, laser.lambda, cavity.space(p).L, cavity.space(p).n);
  end
  result.f_trans = FT_FFT_mirror(result.f_trans,cavity.mirror(transmit),'trans');
  
  % computing field 'leaking out' at the input mirror
  result.f_leak=result.f_circ;
  for p=1:clast-1
    result.f_leak = FT_FFT_propagate(result.f_leak, mygrid, laser.lambda, cavity.space(p).L, cavity.space(p).n);
    result.f_leak = FT_FFT_mirror(result.f_leak,cavity.mirror(p),'front');
  end
  result.f_leak = FT_FFT_propagate(result.f_leak, mygrid, laser.lambda, cavity.space(clast).L, cavity.space(clast).n);
  result.f_leak = result.f_leak * exp(1i*laser.k*result.operating_point);
  result.f_leak = FT_FFT_mirror(result.f_leak,cavity.mirror(clast),'trans');

  % computing reflected field
  result.f_refl = FT_FFT_mirror(result.f_in,cavity.mirror(clast),'back');
  result.f_refl = result.f_refl + result.f_leak;

  return;