%
%---------------------------------------------------------------------  
% function [dfile] = FT_FFT_reorganise_tmp_files(dfile, mygrid);
% 
% Matlab function that reads the temporary files saved by the function
% FT_FFT_precompute_roundtrips and saves them in a new format, 
% required for FT_FFT_cavity_scan.
%
% mygrid (structure):  stores parameters of a 2D grid for FFT computations,
%                      see FT_init_grid.m
%
% dfile (structure):   stored information about the temporary data files.
%                      This functions adds the following fields:
%                      - lines_per_file - number of grid lines stored per file
%                      - number_of_linefiles=filenumber - number of files storing lines
%                      - tmplines - basename for temporary file storing lines
%
%
% Part of the SimTools package
% Andreas Freise, 26.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  
%

% Description: Reads temporary files saved by FT_FFT_precompute_roundtrips
% Description: and saves them in a new format for FT_FFT_cavity_scan
% Keywords: reorganise, tmp, temporary, format, FT_FFT_precompute_roundtrips
% Keywords: FT_FFT_cavity_scan

function [dfile] = FT_FFT_reorganise_tmp_files(dfile, mygrid);

  baseid='FFT_reorganise_files';
  
  %%% TODO replace filenumber by q if tested
  
  dfile.lines_per_file=ceil(mygrid.xpoints/dfile.number_of_files);
  dfile.tmplines=sprintf('tmp_lines_%s_',dfile.datestr);

  total_time=tic;
  fprintf(' === Reorganising %d temporary files...\n',dfile.number_of_files);
  fprintf(' newfile  -   oldfile\n  ');
  fprintf('%4d     -   %4d',1,1);
    
  data.tmpfields = complex(zeros(mygrid.xpoints,mygrid.ypoints,dfile.fields_per_file,'double'));
  tmplines = complex(zeros(dfile.lines_per_file,mygrid.ypoints,dfile.roundtrips,'double'));

  dfile.number_of_linefiles=ceil(mygrid.xpoints/dfile.lines_per_file);
  
  filenumber=0;

  for q=0:dfile.number_of_linefiles-1
    start_data=q*dfile.lines_per_file+1;
    if (q<dfile.number_of_linefiles-1)
      stop_data=(q+1)*dfile.lines_per_file;
    else
      stop_data=mygrid.xpoints;
    end

    for qq=0:dfile.number_of_files-1
      fprintf('\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%4d     -   %4d\n',q+1,qq+1);
      filename=sprintf('%s%04d.mat',dfile.tmpfields,qq);
      data=load(filename);
      start_lines=qq*dfile.fields_per_file+1;
      if (qq<dfile.number_of_files-1)
        stop_lines=start_lines+size(data.tmpfields,3)-1;
      else
        stop_lines=dfile.roundtrips;
      end
    
      tmplines(1:1+stop_data-start_data,:,start_lines:stop_lines)=data.tmpfields(start_data:stop_data,:,1:stop_lines-start_lines+1);
    end
    newfilename=sprintf('%s%04d.mat',dfile.tmplines,filenumber);
    save(newfilename,'tmplines');
    filenumber=filenumber+1;
  end

  if (dfile.number_of_linefiles~=filenumber)
    msgid=[baseid,':filenumber'];
    result=sprintf('Internal error: filenumbers do not match.\n');
    error(msgid,result);
  end
  
  elapsed_time=toc(total_time);
  fprintf([' -> elapsed time: %2.0fh:%2.0fmin:%2.0fsec\n'] ...
          ,fix(elapsed_time/3600),fix(mod(elapsed_time,3600)/60),fix(mod(mod(elapsed_time,3600),60)));
  return

