%
%---------------------------------------------------------------------  
% function [dfile] = FT_FFT_precompute_roundtrips(cavity, in_field, mygrid, lambda, roundtrips, verbose)
% 
% Matlab function that computes fields inside a cavity (given by the
% structure `cavity') and saves the result in temporary data files.
% The function returns a structure `dfile' which stores information
% about the files.
%
% cavity (structure):  cavity parameters and components 
%                      (see FT_init_cavity.m)
% in_field (complex):  2D data of input beam splitude 
% mygrid (structure):  stores parameters of a 2D grid for FFT computations,
%                      see FT_init_grid.m
% roundtrips:          number of roundtrips to be computed
% verbose (integer):   verbose=1 cause the function to print more verbose
%                      status messages
%
% dfile (structure):   stored information about the temporary data files:
%                      - roundtrips 
%                      - number_of_files - number of files saved
%                      - fields_per_file - number of 2D fields stored per file
%                      - datestr - date string to be used in filenames
%                      - tmpfields - basename for temporary file storing fields
%
%
% Part of the SimTools package
% Andreas Freise, 26.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  
%

% Description: Computes fields inside a cavity and saves result in
% Description: temporary data files
% Keywords: field, cavity

function [dfile] = FT_FFT_precompute_roundtrips(cavity, in_field, mygrid, lambda, roundtrips, verbose)

   baseid='precompute_roundtrips';

   clight=299792458.0;

   max_memory=30e6;         % maximum number of doubles in one array

   tic;
   required_memory=mygrid.xpoints*mygrid.ypoints*roundtrips;
   
   dfile.roundtrips=roundtrips;
   dfile.number_of_files=ceil(required_memory/max_memory); % number of temporary data files
   dfile.fields_per_file=ceil(roundtrips/dfile.number_of_files); % max number of fields stored in one file

   fprintf(' === Precomputing %d roundtrips, creating %d temporary data files ...\n',roundtrips, dfile.number_of_files);
   
   dfile.datestr=FT_datestring();
   dfile.tmpfields=sprintf('tmp_fields_%s_',dfile.datestr);


   f_circ = complex(zeros(mygrid.xpoints,mygrid.ypoints,'double'));
   f_round = complex(zeros(mygrid.xpoints,mygrid.ypoints,dfile.fields_per_file,'double'));
   tmpfields=f_round;
   
   clast=cavity.ncomponents;
   
   % move impinging field into cavity
   f_circ = FT_FFT_mirror(in_field,cavity.mirror(clast),'trans');
   % this counts as the first (zeroth) roundtrip
   f_round(:,:,1) = f_circ;

   filenumber=0;
   q=1;
   newline=0;
   elapsed_time=0;
   
   for qq=2:roundtrips
     q=q+1;
     for p=1:clast
       f_circ = FT_FFT_propagate(f_circ, mygrid, lambda, cavity.space(p).L, cavity.space(p).n);
       f_circ = FT_FFT_mirror(f_circ,cavity.mirror(p),'front');
     end
     f_round(:,:,q) = f_circ;
   
     if ((q==dfile.fields_per_file) || (qq==roundtrips))
       filename=sprintf('%s%04d.mat',dfile.tmpfields,filenumber);
       if (verbose)
         newline=1;
         fprintf('\n Saving temporary data file: %s\n',filename);
       end
       if (qq==roundtrips)
         tmpfields=f_round(:,:,1:q);
       else
         tmpfields=f_round;
       end
       save(filename,'tmpfields');
       filenumber=filenumber+1;
       q=0;
     end
     if ((rem(qq,round(roundtrips/100)) == 0) || qq==roundtrips)
       [elapsed_time]=FT_print_progress(elapsed_time,qq,roundtrips, newline);
       tic;
       newline=0;
     end
   end  
   
   dfile.number_of_files=filenumber;
   
   return