%
%---------------------------------------------------------------------  
% function [power] = FT_FFT_cavity_scan(dfile, mygrid, lambda, roundtrips, scan);
% 
% Matlab function that reads the temporary files saved by the functions
% FT_FFT_precompute_roundtrips and FT_FFT_reorganise_files.m
% and computes the circulating power as a function of the cavity tuning.
%
% dfile (structure):    stored information about the temporary data files,
%                       see FT_FFT_reorganise_tmp_files.m
% mygrid (structure):   stores parameters of a 2D grid for FFT computations,
%                       see FT_init_grid.m
% lambda (real):        laser wavelength [m]
% roundtrips (integer): number of roundtrips to use in computation
% scan (real):          vector with microscopic changes in the cavity length [m]
%
% power (real):         vector containing the circulating power for the 
%                       positions given by 'scan' [W]
%
% Part of the SimTools package
% Andreas Freise, 26.12.2009 afreise@googlemail.com
%---------------------------------------------------------------------  
%

% Description: Reads temporary files and computes the circulating power
% Description: as a function of the cavity tuning
% Keywords: circulating, power, cavity, scan

function [power] = FT_FFT_cavity_scan(dfile, mygrid, lambda, roundtrips, scan);

  baseid='FFT_cavity_scan';
  
  if (roundtrips<1 || roundtrips>dfile.roundtrips)
    msgid=[baseid,':checkinarg'];
    result=sprintf('Number of roundtrips is %d but must be >1 and <%d.\n',roundtrips, dfile.roundtrips);
    error(msgid,result);
  end
  
  fprintf(' === Cavity scan: calculating power inside the cavity ...\n')
  total_time=tic;
    
  k=2*pi/lambda;
  points=length(scan);
  power = zeros(1,points,'double');

  f_temp = complex(zeros(dfile.lines_per_file,mygrid.ypoints,'double'));
  data.tmplines = complex(zeros(dfile.lines_per_file,mygrid.ypoints, ...
                                dfile.roundtrips,'double'));
  f_round = complex(zeros(dfile.lines_per_file,mygrid.ypoints, ...
                              dfile.roundtrips,'double'));


  for qq=0:dfile.number_of_linefiles-1
    elapsed_time=0;
    tic;
    filename=sprintf('%s%04d.mat',dfile.tmplines,qq);
    fprintf(' Processing file number %d of %d\n',qq+1,dfile.number_of_linefiles);
    data=load(filename);
    f_round(:)=0.0;
    f_round(1:size(data.tmplines,1),:,:)=data.tmplines;

    for q = 1:points
      f_temp(:)=0.0;
      for qqq = 1:roundtrips
        f_temp = f_temp + f_round(:,:,qqq) * exp(1i*k* scan(q)*qqq);
      end
      power(q) = power(q)+sum(sum(abs(f_temp).^2))*mygrid.xstep*mygrid.ystep;

      if ((rem(q,round(points/100)) == 0) || q==points)
        [elapsed_time]=FT_print_progress(elapsed_time,q,points, 0);
        tic;
      end
       
    end     
  end
    
  elapsed_time=toc(total_time);
  fprintf([' --> total elapsed time: %2.0fh:%2.0fmin:%2.0fsec\n'] ...
          ,fix(elapsed_time/3600),fix(mod(elapsed_time,3600)/60),fix(mod(mod(elapsed_time,3600),60)));

  return