%
%-------------------------------------------------------------------
% function [Mm] = FT_ABCD_mirror(Rc,alpha,n)
%
% Function for Matlab/Octave
% 
% Computes the ABCD Matrix for the reflection of a curved mirror
% or beam splitter 
%
% In the tangential plane:
%    
%             (1                   0 )
% M(:,:,1) =  (2*n/(cos(alpha)*Rc)  1 )
%
% In the sagittal plane as
%    
%            (1                   0 )
% M(:,:,2) = (2*n*cos(alpha)/Rc  1 )
%
% Input:
%   Rc: radius of curvature [m]
%   (use Rc=0 for a flat mirror)
%   alpha: angle of incidence [deg]
%   n: index of refraction
% Output:
%   Mm: ABCD Matrix (2x2x2)
%
% A. Freise 11.02.2010
%-------------------------------------------------------------------
%

% Description: Computes the ABCD Matrix for the reflection of a curved mirror
% Description: or beam splitter
% Keywords: reflection, mirror, splitter, matrix

function [Mm] = FT_ABCD_mirror(Rc,alpha,n)

  Mm=zeros(2,2,2);

  if (length(Rc)==1)
    tmpRc=Rc;
    Rc=zeros(1,2);
    Rc(:)=tmpRc;
  end
  
  if (Rc(1)==0 || isinf(Rc(1)))
    Mm(:,:,1) = [ 1 0 ; 0 1];
  else
    Mm(:,:,1) = [ 1 0 ; -2*n/(cos(alpha/180*pi)*Rc(1)) 1]; 
  end
  
  if (Rc(2)==0 || isinf(Rc(2)))
    Mm(:,:,2) = [ 1 0 ; 0 1];
  else
    Mm(:,:,2) = [ 1 0 ; -2*n*cos(alpha/180*pi)/Rc(2) 1];
  end